<?php

namespace OpenEMR\Tests\Unit\FHIR\SMART\ExternalClinicalDecisionSupport;

use Google\Service\Bigquery\SessionInfo;
use OpenEMR\Common\Auth\OpenIDConnect\Entities\ClientEntity;
use OpenEMR\Common\Auth\OpenIDConnect\Repositories\ClientRepository;
use OpenEMR\Common\Csrf\CsrfUtils;
use OpenEMR\Common\Logging\SystemLogger;
use OpenEMR\FHIR\SMART\ActionUrlBuilder;
use OpenEMR\FHIR\SMART\ExternalClinicalDecisionSupport\PredictiveDSIServiceEntity;
use OpenEMR\FHIR\SMART\ExternalClinicalDecisionSupport\RouteController;
use OpenEMR\Services\DecisionSupportInterventionService;
use PHPUnit\Framework\Constraint\IsEqual;
use PHPUnit\Framework\TestCase;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Session\Session;
use Symfony\Component\HttpFoundation\Session\SessionInterface;
use Symfony\Component\HttpFoundation\Session\Storage\MockArraySessionStorage;
use Twig\Environment;

class RouteControllerTest extends TestCase
{
    protected RouteController $controller;
    protected \PHPUnit\Framework\MockObject\MockObject $mockRepository;
    protected \PHPUnit\Framework\MockObject\MockObject $mockLogger;

    protected \PHPUnit\Framework\MockObject\MockObject $mockTwig;

    protected \PHPUnit\Framework\MockObject\MockObject $actionUrlBuilder;

    protected \PHPUnit\Framework\MockObject\MockObject $dsiService;

    private SessionInterface $session;

    protected function setUp(): void
    {
        parent::setUp(); // TODO: Change the autogenerated stub
        $this->mockRepository = $this->createMock(ClientRepository::class);
        $this->mockLogger = $this->createMock(SystemLogger::class);
        $this->mockTwig = $this->createMock(Environment::class);
        $this->actionUrlBuilder = $this->createMock(ActionUrlBuilder::class);
        $this->dsiService = $this->createMock(DecisionSupportInterventionService::class);
        $this->dsiService->method('getClientRepository')->willReturn($this->mockRepository);
        $this->session = new Session(new MockArraySessionStorage());
        CsrfUtils::setupCsrfKey($this->session);
        $this->controller = new RouteController($this->session, $this->mockRepository, $this->mockLogger, $this->mockTwig, $this->actionUrlBuilder, $this->dsiService);
    }

    public function testSupportsRequest(): void
    {
        $request = new Request(['action' => 'external-cdr/']);
        $this->assertTrue($this->controller->supportsRequest($request));

        $request = new Request(['action' => 'external-cd/']);
        $this->assertFalse($this->controller->supportsRequest($request));

        $request = new Request(['action' => 'external-cdr']);
        $this->assertTrue($this->controller->supportsRequest($request));

        $request = new Request(['action' => 'external-cdr/1']);
        $this->assertTrue($this->controller->supportsRequest($request));

        $request = new Request(['action' => 'external-cdr/1/edit/']);
        $this->assertTrue($this->controller->supportsRequest($request));

        $request = new Request(['action' => 'external-cd/1']);
        $this->assertFalse($this->controller->supportsRequest($request));
    }

    public function testParseRequest(): void
    {
        $request = new Request(['action' => 'external-cdr/']);
        $result = $this->controller->parseRequest($request);
        $this->assertEquals('external-cdr', $result['mainAction']);
        $this->assertEmpty($result['mainActionChild']);
        $this->assertEmpty($result['subAction']);

        $request = new Request(['action' => 'external-cdr/list/']);
        $result = $this->controller->parseRequest($request);
        $this->assertEquals('external-cdr', $result['mainAction']);
        $this->assertEquals('list', $result['mainActionChild']);
        $this->assertEmpty($result['subAction']);

        $request = new Request(['action' => 'external-cdr/1/edit/']);
        $result = $this->controller->parseRequest($request);
        $this->assertEquals('external-cdr', $result['mainAction']);
        $this->assertEquals('1', $result['mainActionChild']);
        $this->assertEquals('edit', $result['subAction']);
    }

    public function testDispatch(): void
    {
        $id = "1";
        $this->setupDSIServiceForClientEntity($id, "Test Client");
        $request = new Request(['action' => 'external-cdr/edit/' . $id]);
        $this->assertTrue($this->controller->supportsRequest($request));
        $response = $this->controller->dispatch($request);
        $this->assertEquals(200, $response->getStatusCode());
    }

    private function setupDSIServiceForClientEntity(string $id, string $name)
    {
        $client = new ClientEntity();
        $client->setIdentifier($id);
        $client->setName($name);
        $dsiService = new PredictiveDSIServiceEntity($client);
        $this->dsiService->method("getService")->willReturn($dsiService);
    }
    public function testSaveAction(): void
    {
        $id = "1";
//        CsrfUtils::setupCsrfKey(); // setup the key
        $this->setupDSIServiceForClientEntity($id, "Test Client");
        $this->session->set('authUserID', 1);
        $request = new Request(['action' => 'external-cdr/save/' . $id]);
        $request->request->set("predictive_details_developer", "OpenEMR");
        $request->request->set("predictive_details_funding", "Donations");
        $request->request->set('_token', CsrfUtils::collectCsrfToken('default', $this->session));

        $this->dsiService->expects($this->once())->method("updatePredictiveDSIAttributes");

        // $saveResponseUrl = $this->actionUrlBuilder->buildUrl([self::EXTERNAL_CDR_ACTION, 'edit', $service->getId()], ['queryParams' => ['status' => $status]]);
        $this->actionUrlBuilder->expects($this->once())->method("buildUrl")
            ->with(
                $this->identicalTo([RouteController::EXTERNAL_CDR_ACTION, 'edit', $id]),
                $this->identicalTo(['queryParams' => ['status' => 'success']])
            );

        // smoke test
        $this->assertTrue($this->controller->supportsRequest($request), "Request should be supported");
        $response = $this->controller->dispatch($request);
        $this->assertEquals(302, $response->getStatusCode());
    }
}
