<?php

/*
 * FhirLocationServiceTest.php
 * @package openemr
 * @link      http://www.open-emr.org
 * @author    Stephen Nielson <snielson@discoverandchange.com>
 * @copyright Copyright (c) 2025 Stephen Nielson <snielson@discoverandchange.com>
 * @copyright Public Domain for the pieces that were generated by Claude.AI
 * @license   https://github.com/openemr/openemr/blob/master/LICENSE GNU General Public License 3
 */

namespace OpenEMR\Tests\Services\FHIR;

use OpenEMR\FHIR\Export\ExportJob;
use OpenEMR\FHIR\Export\ExportStreamWriter;
use OpenEMR\FHIR\R4\FHIRDomainResource\FHIRLocation;
use OpenEMR\Services\FHIR\FhirLocationService;
use OpenEMR\Services\LocationService;
use OpenEMR\Services\Search\CompositeSearchField;
use OpenEMR\Services\Search\TokenSearchField;
use OpenEMR\Services\Search\TokenSearchValue;
use OpenEMR\Validators\ProcessingResult;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;
use Ramsey\Uuid\Uuid;
use Symfony\Component\HttpFoundation\Session\Session;
use Symfony\Component\HttpFoundation\Session\Storage\MockFileSessionStorage;

class FhirLocationServiceTest extends TestCase
{
    // AI Generated Test Class for FhirLocationService
    /**
     * @var FhirLocationService
     */
    private FhirLocationService $fhirLocationService;

    /**
     * @var LocationService|MockObject
     */
    private MockObject $mockLocationService;

    /**
     * @var ExportStreamWriter|MockObject
     */
    private MockObject $mockStreamWriter;

    private Session $mockSession;

    protected function setUp(): void
    {
        parent::setUp();

        $this->mockLocationService = $this->createMock(LocationService::class);
        $this->mockStreamWriter = $this->createMock(ExportStreamWriter::class);

        $this->fhirLocationService = new FhirLocationService();
        $this->fhirLocationService->setLocationService($this->mockLocationService);
        $this->mockSession = new Session(new MockFileSessionStorage());
        $this->mockSession->set("authUser", "admin"); // admin user
        $this->fhirLocationService->setSession($this->mockSession);
    }

    /**
     * Test that Group export only includes patient locations for patients in the group
     * and excludes patient locations for patients not in the group
     */
    public function testExportGroupOperationIncludesOnlyGroupPatientLocations(): void
    {
        // AI-generated test setup - START
        // Patient UUIDs
        $groupPatientUuid1 = Uuid::uuid4()->toString();
        $groupPatientUuid2 = Uuid::uuid4()->toString();
        $nonGroupPatientUuid = Uuid::uuid4()->toString();

        // Location UUIDs
        $facilityLocationUuid = Uuid::uuid4()->toString();
        $groupPatient1LocationUuid = Uuid::uuid4()->toString();
        $groupPatient2LocationUuid = Uuid::uuid4()->toString();
        $nonGroupPatientLocationUuid = Uuid::uuid4()->toString();

        // Create mock export job for Group operation
        $mockJob = new ExportJob();
        $mockJob->setExportType(ExportJob::EXPORT_OPERATION_GROUP);
        $mockJob->setResourceIncludeTime(\DateTime::createFromFormat('Y-m-d H:i:s', '2023-01-01 01:00:00'));
        $mockJob->setPatientUuidsToExport([$groupPatientUuid1, $groupPatientUuid2]);
        // AI-generated test setup - END

        // Mock location data that would be returned from LocationService
        $locationData = [
            // Facility location - should always be included
            [
                'uuid' => $facilityLocationUuid,
                'name' => 'Main Clinic',
                'type' => 'facility',
                'last_updated' => '2023-01-15 10:00:00'
            ],
            // Patient location for group member - should be included
            [
                'uuid' => $groupPatient1LocationUuid,
                'name' => 'Patient Home Address 1',
                'type' => 'patient',
                'table_uuid' => $groupPatientUuid1,
                'last_updated' => '2023-01-15 11:00:00'
            ],
            // Patient location for group member - should be included
            [
                'uuid' => $groupPatient2LocationUuid,
                'name' => 'Patient Home Address 2',
                'type' => 'patient',
                'table_uuid' => $groupPatientUuid2,
                'last_updated' => '2023-01-15 12:00:00'
            ]
            // Note: Non-group patient location should not be returned by LocationService
            // due to the search filtering applied
        ];

        $processingResult = new ProcessingResult();
        $processingResult->setData($locationData);

        // AI-generated mock expectations - START
        // Mock LocationService to return filtered results
        $this->mockLocationService
            ->expects($this->once())
            ->method('getAll')
            ->with($this->callback(function ($searchParams) use ($groupPatientUuid1, $groupPatientUuid2) {
                // Verify that the search parameters include patient filtering for group members
                $this->assertArrayHasKey('patient-facility-type', $searchParams);
                $this->assertArrayHasKey('last_updated', $searchParams);
                $this->assertInstanceOf(CompositeSearchField::class, $searchParams['patient-facility-type']);
                $this->assertPatientFacilityTypeSearchField($searchParams['patient-facility-type'], [$groupPatientUuid1, $groupPatientUuid2]);

                return true;
            }))
            ->willReturn($processingResult);

        // Mock StreamWriter to capture appended resources
        $appendedResources = [];
        $this->mockStreamWriter
            ->expects($this->exactly(3)) // Facility + 2 patient locations
            ->method('append')
            ->willReturnCallback(function ($resource) use (&$appendedResources): void {
                $appendedResources[] = $resource;
            });
        // AI-generated mock expectations - END

        // Execute the export
        $this->fhirLocationService->export($this->mockStreamWriter, $mockJob);

        // Verify the correct number of resources were exported
        $this->assertCount(3, $appendedResources, 'Should export exactly 3 locations: 1 facility + 2 group patient locations');

        // Verify each exported resource
        $exportedUuids = [];
        foreach ($appendedResources as $resource) {
            $this->assertInstanceOf(FHIRLocation::class, $resource);
            $exportedUuids[] = $resource->getId()->getValue();
        }

        // Assert that the correct locations were exported
        $this->assertContains($facilityLocationUuid, $exportedUuids, 'Facility location should be included');
        $this->assertContains($groupPatient1LocationUuid, $exportedUuids, 'Group patient 1 location should be included');
        $this->assertContains($groupPatient2LocationUuid, $exportedUuids, 'Group patient 2 location should be included');

        // Most importantly, verify that non-group patient location was NOT exported
        $this->assertNotContains($nonGroupPatientLocationUuid, $exportedUuids, 'Non-group patient location should NOT be included');
    }

    /**
     * Test that Group export with empty patient list exports only facility locations
     */
    public function testExportGroupOperationWithEmptyPatientListExportsOnlyFacilities(): void
    {
        // Create mock export job with empty patient list
        $mockJob = $this->createMock(ExportJob::class);
        $mockJob->method('getExportType')->willReturn(ExportJob::EXPORT_OPERATION_GROUP);
        $mockJob->method('getPatientUuidsToExport')->willReturn([]); // Empty patient list

        // Mock StreamWriter - should not append anything
        $this->mockStreamWriter
            ->expects($this->never())
            ->method('append');

        // Execute the export - should return early with no processing
        $this->fhirLocationService->export($this->mockStreamWriter, $mockJob);
    }

    /**
     * Test that System export includes all locations regardless of patient filtering
     */
    public function testExportSystemOperationIncludesAllLocations(): void
    {
        // AI-generated test data - START
        $facilityLocationUuid = 'facility-location-1';
        $patientLocation1Uuid = 'patient-location-1';
        $patientLocation2Uuid = 'patient-location-2';

        // Create mock export job for System operation
        $mockJob = new ExportJob();
        $mockJob->setExportType(ExportJob::EXPORT_OPERATION_SYSTEM);
        $mockJob->setResourceIncludeTime(\DateTime::createFromFormat('Y-m-d H:i:s', '2023-01-01 01:00:00'));
        // AI-generated test data - END

        // Mock location data - all locations should be included for system export
        $locationData = [
            [
                'uuid' => $facilityLocationUuid,
                'name' => 'Main Clinic',
                'type' => 'facility',
                'last_updated' => '2023-01-15 10:00:00'
            ],
            [
                'uuid' => $patientLocation1Uuid,
                'name' => 'Patient Home 1',
                'type' => 'patient',
                'last_updated' => '2023-01-15 11:00:00'
            ],
            [
                'uuid' => $patientLocation2Uuid,
                'name' => 'Patient Home 2',
                'type' => 'patient',
                'last_updated' => '2023-01-15 12:00:00'
            ]
        ];

        $processingResult = new ProcessingResult();
        $processingResult->setData($locationData);

        // Mock LocationService for system export (no patient filtering)
        $this->mockLocationService
            ->expects($this->once())
            ->method('getAll')
            ->with($this->callback(function ($searchParams) {
                // For system export, should only have _lastUpdated (which turns into oemr last_updated field), no patient filtering
                $this->assertArrayHasKey('last_updated', $searchParams);
                $this->assertArrayNotHasKey('patient-facility-type', $searchParams);
                return true;
            }))
            ->willReturn($processingResult);

        // Mock StreamWriter
        $this->mockStreamWriter
            ->expects($this->exactly(3)) // All 3 locations should be exported
            ->method('append');

        // Execute the export
        $this->fhirLocationService->export($this->mockStreamWriter, $mockJob);
    }

    /**
     * Test the patient context search field creation for group filtering
     */
    public function testGetPatientContextSearchFieldForPatientUuids(): void
    {
        $patientUuids = [Uuid::uuid4()->toString(), Uuid::uuid4()->toString()];

        // Use reflection to access the private method
        $reflection = new \ReflectionClass($this->fhirLocationService);
        $method = $reflection->getMethod('getPatientContextSearchFieldForPatientUuids');

        $searchField = $method->invoke($this->fhirLocationService, $patientUuids);

        // Verify the search field structure
        $this->assertInstanceOf(CompositeSearchField::class, $searchField);
        $this->assertEquals('patient-facility-type', $searchField->getName());
        $this->assertFalse($searchField->isAnd()); // Should be OR condition

        // Verify it has 2 children (facility-type and patient-type)
        $children = $searchField->getChildren();
        $this->assertCount(2, $children);

        // Find the patient-type and facility-type children
        $patientType = null;
        $facilityType = null;
        foreach ($children as $child) {
            if ($child->getName() === 'patient-type') {
                $patientType = $child;
            } elseif ($child->getName() === 'facility-type') {
                $facilityType = $child;
            }
        }

        $this->assertNotNull($patientType, 'Should have patient-type search field');
        $this->assertNotNull($facilityType, 'Should have facility-type search field');

        // Verify patient-type has table_uuid and type filters
        $patientChildren = $patientType->getChildren();
        $this->assertCount(2, $patientChildren); // table_uuid and type

        // Verify facility-type has type filter
        $facilityChildren = $facilityType->getChildren();
        $this->assertCount(1, $facilityChildren); // just type
    }

    /**
     * Helper method to create a mock FHIR Location resource
     */
    private function createMockFhirLocation(string $uuid, string $name, string $type): FHIRLocation
    {
        $fhirLocation = $this->createMock(FHIRLocation::class);

        // Mock the ID
        $mockId = $this->createMock(\OpenEMR\FHIR\R4\FHIRElement\FHIRId::class);
        $mockId->method('getValue')->willReturn($uuid);
        $fhirLocation->method('getId')->willReturn($mockId);

        // Mock the name
        $fhirLocation->method('getName')->willReturn($name);

        return $fhirLocation;
    }
    // end AI Generated Test Class for FhirLocationService
    private function assertPatientFacilityTypeSearchField(CompositeSearchField $patientFacilityField, array $expectedPatientUuids): void
    {
        // Verify the patient-facility-type search field structure
        // verify the search has the patient ids
        $this->assertCount(2, $patientFacilityField->getChildren());
        $facilityTypeComposite = $patientFacilityField->getChildren()[0];
        $this->assertInstanceOf(CompositeSearchField::class, $facilityTypeComposite);
        $this->assertEquals('facility-type', $facilityTypeComposite->getName());
        $this->assertCount(1, $facilityTypeComposite->getChildren(), "facility-type composite should have 1 child (type)");
        $facilityTypeField = $facilityTypeComposite->getChildren()[0];
        $this->assertEquals('type', $facilityTypeField->getName());

        $patientTypeComposite = $patientFacilityField->getChildren()[1];
        $this->assertEquals('patient-type', $patientTypeComposite->getName(), "First child of patient-facility-type should be patient-type");
        $this->assertInstanceOf(CompositeSearchField::class, $patientTypeComposite);
        $this->assertCount(2, $patientTypeComposite->getChildren(), "patient-type composite should have 2 children (table_uuid and type)");
        $patientUuidField = $patientTypeComposite->getChildren()[0];
        $patientTypeField = $patientTypeComposite->getChildren()[1];
        $this->assertEquals('table_uuid', $patientUuidField->getName());
        $this->assertEquals('type', $patientTypeField->getName());
        $this->assertInstanceOf(TokenSearchField::class, $patientUuidField);
        $patientUuids = $patientUuidField->transformValues(fn($value): string => $value instanceof TokenSearchValue ? $value->getHumanReadableCode() : (string)$value);
        $this->assertEquals($expectedPatientUuids, $patientUuids);
    }
}
