<?php

/*
 * FhirQuestionnaireResponseRestControllerTest.php
 * @package openemr
 * @link      http://www.open-emr.org
 * @author    Stephen Nielson <snielson@discoverandchange.com>
 * @copyright Copyright (c) 2025 Stephen Nielson <snielson@discoverandchange.com>
 * @license   https://github.com/openemr/openemr/blob/master/LICENSE GNU General Public License 3
 */

namespace OpenEMR\Tests\RestControllers\FHIR;

use OpenEMR\Common\Database\QueryUtils;
use OpenEMR\Common\Http\HttpRestRequest;
use OpenEMR\Common\Uuid\UuidRegistry;
use OpenEMR\RestControllers\FHIR\FhirQuestionnaireResponseRestController;
use OpenEMR\Services\FHIR\FhirQuestionnaireResponseService;
use OpenEMR\Services\FHIR\QuestionnaireResponse\FhirQuestionnaireResponseFormService;
use OpenEMR\Services\PatientService;
use OpenEMR\Services\QuestionnaireResponseService;
use OpenEMR\Services\QuestionnaireService;
use PHPUnit\Framework\TestCase;
use PHPUnit\Framework\MockObject\Exception;
use Symfony\Component\EventDispatcher\EventDispatcher;
use Symfony\Component\HttpFoundation\Response;

class FhirQuestionnaireResponseRestControllerIntegrationTest extends TestCase
{
    private array $originalSession;

    private string $questionnaireTemplate;

    private string $questionnaireResponseTemplate;

    const QUESTIONNAIRE_NAME_1 = "PHQ-9 Example";
    const QUESTIONNAIRE_NAME_2 = "PHQ-9";
    protected function setUp(): void
    {
        parent::setUp(); // TODO: Change the autogenerated stub
        // TODO: When we have a session storage we can mock we want to get away from
        // this session behavior
        $this->originalSession = $_SESSION;
        $this->questionnaireTemplate = file_get_contents(__DIR__ . '/../../data/Services/FHIR/QuestionnaireResponse/Questionnaire-phq-9-example.json');
        $this->questionnaireResponseTemplate = file_get_contents(__DIR__ . '/../../data/Services/FHIR/QuestionnaireResponse/QuestionnaireResponse-phq-9-example.json');
    }

    protected function tearDown(): void
    {
        parent::tearDown();
        $_SESSION = $this->originalSession; // restore the session sadly.
        // need to clean up the database records
        $name = self::QUESTIONNAIRE_NAME_1;
        $name2 = self::QUESTIONNAIRE_NAME_2;
        QueryUtils::sqlStatementThrowException("DELETE FROM uuid_registry WHERE uuid IN (select uuid FROM questionnaire_response WHERE questionnaire_foreign_id IN (select id FROM questionnaire_repository WHERE name =? OR name = ?))", [$name, $name2]);
        QueryUtils::sqlStatementThrowException("DELETE FROM questionnaire_response WHERE questionnaire_foreign_id IN (select id FROM questionnaire_repository WHERE name =? OR name = ?)", [$name, $name2]);
        QueryUtils::sqlStatementThrowException("DELETE FROM uuid_registry WHERE uuid IN (select uuid FROM questionnaire_repository WHERE name =? OR name = ?)", [$name, $name2]);
        QueryUtils::sqlStatementThrowException("DELETE FROM questionnaire_repository WHERE name = ? OR name = ?", [$name, $name2]);
        QueryUtils::sqlStatementThrowException("DELETE FROM patient_data WHERE pubpid LIKE 'phpunit-test-'", []);
    }

    public function testCreate(): void
    {
        $this->markTestIncomplete("update is not exposed yet, so leaving test as incomplete until we choose to expose it");
    }

    public function testOne(): void
    {
        $_SESSION['authUserID'] = QueryUtils::fetchSingleValue('select id FROM users ORDER BY id LIMIT 1', 'id');
        $patientService = new PatientService();
        $result = $patientService->insert([
            'DOB' => '2000-01-01'
            ,'fname' => 'Test user'
            ,'lname' => 'Test User'
            ,'sex' => 'Male'
            ,'pubpid' => 'phpunit-test-' . date("Y-m-d-H-i-s")
        ]);
        $this->assertTrue($result->isValid(), "Failed creating patient_data record");
        $patientIds = $result->getFirstDataResult();
        $questionnaireResponseUuid = $this->createQuestionnaireResponseForQuestionnaire($patientIds['pid'], $patientIds['uuid'], "PHQ-9 Example", null);
        $restRequest = $this->createMock(HttpRestRequest::class);
        $dispatcher = new EventDispatcher();
        $fhirService = new FhirQuestionnaireResponseService();
        $fhirService->addMappedService(new FhirQuestionnaireResponseFormService());
        $fhirService->setEventDispatcher($dispatcher);
        $controller = new FhirQuestionnaireResponseRestController($fhirService);
        $response = $controller->one($restRequest, $questionnaireResponseUuid);
        $this->assertEquals(Response::HTTP_OK, $response->getStatusCode());
        $contents = $response->getBody()->getContents();
        $this->assertNotEmpty($contents, "Response body should not be empty");
        $jsonBundle = json_decode($contents, true);
        $this->assertArrayHasKey("resourceType", $jsonBundle, "Response should contain 'resourceType' key");
        $this->assertEquals('QuestionnaireResponse', $jsonBundle['resourceType'], "Response 'resourceType' should be 'Bundle'");
        $this->assertEquals($questionnaireResponseUuid, $jsonBundle['id']);
    }

    public function testList(): void
    {
        $_SESSION['authUserID'] = QueryUtils::fetchSingleValue('select id FROM users ORDER BY id LIMIT 1', 'id');
        $patientService = new PatientService();
        $result = $patientService->insert([
            'DOB' => '2000-01-01'
            ,'fname' => 'Test user'
            ,'lname' => 'Test User'
            ,'sex' => 'Male'
            ,'pubpid' => 'phpunit-test-' . date("Y-m-d-H-i-s")
        ]);
        $this->assertTrue($result->isValid(), "Failed creating patient_data record");
        $patientIds = $result->getFirstDataResult();
        $questionnaireResponseUuid = $this->createQuestionnaireResponseForQuestionnaire($patientIds['pid'], $patientIds['uuid'], "PHQ-9 Example", null);
        $restRequest = $this->createMock(HttpRestRequest::class);
        $fhirService = new FhirQuestionnaireResponseService();
        $fhirService->addMappedService(new FhirQuestionnaireResponseFormService());
        $dispatcher = new EventDispatcher();
        $fhirService->setEventDispatcher($dispatcher);
        $controller = new FhirQuestionnaireResponseRestController($fhirService);
        $response = $controller->list($restRequest);


        $this->assertEquals(Response::HTTP_OK, $response->getStatusCode());
        $contents = $response->getBody()->getContents();
        $this->assertNotEmpty($contents, "Response body should not be empty");
        $jsonBundle = json_decode($contents, true);
        $this->assertArrayHasKey("resourceType", $jsonBundle, "Response should contain 'resourceType' key");
        $this->assertEquals('Bundle', $jsonBundle['resourceType'], "Response 'resourceType' should be 'Bundle'");
        $this->assertArrayHasKey("entry", $jsonBundle, "Response should contain 'entry' key");
        $this->assertCount(1, $jsonBundle['entry'], "Response 'entry' should contain one item");
        $this->assertArrayHasKey("resource", $jsonBundle['entry'][0], "'entry' should contain 'resource' key");
        $this->assertArrayHasKey("resourceType", $jsonBundle['entry'][0]['resource'], "Response should contain 'resourceType' key");
        $this->assertEquals("QuestionnaireResponse", $jsonBundle['entry'][0]['resource']['resourceType'], "Response 'resourceType' should be 'QuestionnaireResponse'");
        $this->assertEquals($questionnaireResponseUuid, $jsonBundle['entry'][0]['resource']['id']);
    }
    /**
     * @param string $name
     * @return string
     * @throws Exception
     */
    private function createQuestionnaireWithName(string $name): string
    {
        $questionnaireContents = $this->questionnaireTemplate;
        $questionnaireAsJson = json_decode($questionnaireContents, true);
        $questionnaireAsJson['name'] = $questionnaireAsJson['title'] = $name;
        $questionnaireService = new QuestionnaireService();
        $insertId = $questionnaireService->saveQuestionnaireResource($questionnaireAsJson);
        // need to grab the record because we don't get ANY kind of uuid from this
        $binUuid = QueryUtils::fetchSingleValue('SELECT uuid FROM questionnaire_repository WHERE id = ?', 'uuid', [$insertId]);
        $this->assertNotEmpty($binUuid, "Questionnaire uuid resource should be returned");
        return UuidRegistry::uuidToString($binUuid);
    }

    private function createQuestionnaireResponseForQuestionnaire(int $pid, string $puuid, string $name, ?int $encounter): string
    {
        $this->createQuestionnaireWithName($name);
        $questionnaireResponseService = new QuestionnaireResponseService();
        // TODO: @adunsulag do we populate the puuid here?
        $jsonObj = json_decode($this->questionnaireTemplate, true);
        $jsonObj['id'] = null; // clear out the id so the service will save
        $result = $questionnaireResponseService->saveQuestionnaireResponse($jsonObj, $pid, $encounter, null, null, $this->questionnaireTemplate);
        $qrId = $result['id'];
        // grab the uuid and return it
        $binUuid = QueryUtils::fetchSingleValue('SELECT uuid FROM questionnaire_response WHERE id = ?', 'uuid', [$qrId]);
        $this->assertNotEmpty($binUuid, "questionnaire_response.uuid should be returned");
        return UuidRegistry::uuidToString($binUuid);
    }
}
