<?php

/*
 * SMARTSessionTokenContextIntegrationTest.php
 * @package openemr
 * @link      http://www.open-emr.org
 * @author    Stephen Nielson <snielson@discoverandchange.com>
 * @copyright Copyright (c) 2025 Stephen Nielson <snielson@discoverandchange.com>
 * @license   https://github.com/openemr/openemr/blob/master/LICENSE GNU General Public License 3
 */

/**
 * Integration tests for SMART on FHIR session token context handling between
 * SMARTSessionTokenContextBuilder and IdTokenSMARTResponse classes.
 *
 * AI-GENERATED CODE: This integration test suite was generated using Claude AI assistant
 * on August 15, 2025 to test the complete workflow of SMART launch contexts, including EHR launch,
 * standalone launch, context preservation during token refresh, error handling, and logging integration.
 *
 * This code is released into the public domain as it was entirely generated by Claude AI.
 *
 * @package   OpenEMR
 * @link      http://www.open-emr.org
 * @author    Stephen Nielson <snielson@discoverandchange.com> (AI-assisted)
 * @license   https://github.com/openemr/openemr/blob/master/LICENSE GNU General Public License 3
 *
 */

namespace OpenEMR\Tests\Common\Auth\OpenIDConnect;

use League\OAuth2\Server\CryptKey;
use League\OAuth2\Server\Entities\AccessTokenEntityInterface;
use League\OAuth2\Server\Entities\ClientEntityInterface;
use League\OAuth2\Server\Entities\ScopeEntityInterface;
use League\OAuth2\Server\Entities\UserEntityInterface;
use League\OAuth2\Server\Exception\OAuthServerException;
use OpenEMR\Common\Auth\OpenIDConnect\Entities\UserEntity;
use OpenEMR\Common\Auth\OpenIDConnect\IdTokenSMARTResponse;
use OpenEMR\Common\Auth\OpenIDConnect\SMARTSessionTokenContextBuilder;
use OpenEMR\Common\Http\Psr17Factory;
use OpenEMR\Common\Logging\SystemLogger;
use OpenEMR\Core\OEGlobalsBag;
use OpenEMR\FHIR\Config\ServerConfig;
use OpenEMR\FHIR\SMART\SmartLaunchController;
use OpenEMR\FHIR\SMART\SMARTLaunchToken;
use OpenIDConnectServer\ClaimExtractor;
use OpenIDConnectServer\Entities\ClaimSetInterface;
use OpenIDConnectServer\Repositories\IdentityProviderInterface;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;
use Psr\Http\Message\ResponseInterface;
use Symfony\Component\HttpFoundation\Session\Session;
use Symfony\Component\HttpFoundation\Session\Storage\MockArraySessionStorage;

class SMARTSessionTokenContextIntegrationTest extends TestCase
{
    private Session $session;
    private SMARTSessionTokenContextBuilder $contextBuilder;
    private IdTokenSMARTResponse $idTokenResponse;
    private OEGlobalsBag $globalsBag;
    private MockObject $identityProvider;
    private MockObject $claimExtractor;
    private MockObject $logger;

    const KEY_PATH_PRIVATE = __DIR__ . '/../../../data/Unit/Common/Auth/Grant/openemr-rsa384-private.key';
    private OEGlobalsBag $oldGlobals;

    private ServerConfig $serverConfig;

    protected function setUp(): void
    {
        $this->oldGlobals = new OEGlobalsBag(
            [
                'site_addr_oath' => $GLOBALS['site_addr_oath'] ?? null,
                'web_root' => $GLOBALS['web_root'] ?? null,
            ]
        );
        // Set up global variables that are referenced
        $this->globalsBag = new OEGlobalsBag([
            'site_addr_oath' => 'https://example.com',
            'web_root' => '/openemr',
        ]);
        foreach ($this->globalsBag->all() as $key => $value) {
            $GLOBALS[$key] = $value;
        }
        $this->serverConfig = new ServerConfig();
        // Use a real session with mock storage for integration testing
        $this->session = new Session(new MockArraySessionStorage());
        $this->session->set('site_id', 'default');
        $this->contextBuilder = new SMARTSessionTokenContextBuilder($this->serverConfig, $this->session);


        $mockUserEntity = $this->createMock(UserEntity::class);
        $mockUserEntity->method('getIdentifier')
            ->willReturn('test-user-id');
        $mockUserEntity->method('getClaims')
            ->willReturn([]);
        $this->identityProvider = $this->createMock(IdentityProviderInterface::class);
        $this->identityProvider->method('getUserEntityByIdentifier')
            ->willReturn($mockUserEntity);
        $this->claimExtractor = $this->createMock(ClaimExtractor::class);
        $this->claimExtractor->method('extract')
            ->willReturn([]);
        $this->logger = $this->createMock(SystemLogger::class);

        $this->idTokenResponse = new IdTokenSMARTResponse(
            $this->globalsBag,
            $this->session,
            $this->identityProvider,
            $this->claimExtractor,
            $this->contextBuilder
        );

        $this->contextBuilder->setSystemLogger($this->logger);
        $this->idTokenResponse->setSystemLogger($this->logger);

        $cryptKey = $this->createMock(CryptKey::class);
        $cryptKey->method('getKeyPath')
            ->willReturn(self::KEY_PATH_PRIVATE);

        $this->idTokenResponse->setPrivateKey($cryptKey);
    }

    protected function tearDown(): void
    {
        foreach ($this->oldGlobals->all() as $key => $value) {
            if ($value === null) {
                unset($GLOBALS[$key]);
            } else {
                $GLOBALS[$key] = $value;
            }
        }
    }

    public function testEHRLaunchContextIntegration(): void
    {
        // Create a SMART launch token
        $launchToken = new SMARTLaunchToken();
        $launchToken->setPatient('test-patient-uuid');
        $launchToken->setEncounter('test-encounter-uuid');
        $launchToken->setIntent(SMARTLaunchToken::INTENT_PATIENT_DEMOGRAPHICS_DIALOG);
        $launchToken->setAppointmentUuid('test-appointment-uuid');

        // Set the serialized token in session
        $this->session->set('launch', $launchToken->serialize());

        // Create EHR launch scopes
        $scopes = [$this->createMockScope(SmartLaunchController::CLIENT_APP_REQUIRED_LAUNCH_SCOPE)];

        // Test context builder
        $context = $this->contextBuilder->getContextForScopes($scopes);

        $this->assertArrayHasKey('patient', $context);
        $this->assertArrayHasKey('encounter', $context);
        $this->assertArrayHasKey('intent', $context);
        $this->assertArrayHasKey('fhirContext', $context);
        $this->assertArrayHasKey('smart_style_url', $context);
        $this->assertArrayHasKey('need_patient_banner', $context);

        $this->assertEquals('test-patient-uuid', $context['patient']);
        $this->assertEquals('test-encounter-uuid', $context['encounter']);
        $this->assertEquals(SMARTLaunchToken::INTENT_PATIENT_DEMOGRAPHICS_DIALOG, $context['intent']);
        $this->assertFalse($context['need_patient_banner']);
        $this->assertIsArray($context['fhirContext']);
        $this->assertStringContainsString('smart-style', $context['smart_style_url']);

        // Now test integration with IdTokenSMARTResponse
        $this->idTokenResponse->setContextForNewTokens($context);

        $accessToken = $this->createMockAccessTokenWithScopes($scopes);
        $this->setAccessToken($accessToken);

        // Use reflection to test the protected getExtraParams method
        $reflection = new \ReflectionClass($this->idTokenResponse);
        $method = $reflection->getMethod('getExtraParams');

        $extraParams = $method->invoke($this->idTokenResponse, $accessToken);

        $this->assertIsArray($extraParams);
        $this->assertArrayHasKey('patient', $extraParams);
        $this->assertArrayHasKey('encounter', $extraParams);
        $this->assertArrayHasKey('intent', $extraParams);
        $this->assertArrayHasKey('scope', $extraParams);

        $this->assertEquals('test-patient-uuid', $extraParams['patient']);
        $this->assertEquals('test-encounter-uuid', $extraParams['encounter']);
        $this->assertEquals(SMARTLaunchToken::INTENT_PATIENT_DEMOGRAPHICS_DIALOG, $extraParams['intent']);
    }

    public function testStandaloneLaunchContextIntegration(): void
    {
        // Set patient UUID in session for standalone launch
        $this->session->set('puuid', 'standalone-patient-uuid');

        // Create standalone launch scopes
        $scopes = [$this->createMockScope(SmartLaunchController::CLIENT_APP_STANDALONE_LAUNCH_SCOPE)];

        // Test context builder
        $context = $this->contextBuilder->getContextForScopes($scopes);

        $this->assertIsArray($context);
        $this->assertArrayHasKey('patient', $context);
        $this->assertArrayHasKey('need_patient_banner', $context);
        $this->assertArrayHasKey('smart_style_url', $context);

        $this->assertEquals('standalone-patient-uuid', $context['patient']);
        $this->assertTrue($context['need_patient_banner']);
        $this->assertStringContainsString('smart-style', $context['smart_style_url']);

        // Test integration with IdTokenSMARTResponse
        $this->idTokenResponse->setContextForNewTokens($context);

        $accessToken = $this->createMockAccessTokenWithScopes($scopes);
        $this->setAccessToken($accessToken);

        // Use reflection to test the protected getExtraParams method
        $reflection = new \ReflectionClass($this->idTokenResponse);
        $method = $reflection->getMethod('getExtraParams');

        $extraParams = $method->invoke($this->idTokenResponse, $accessToken);

        $this->assertIsArray($extraParams);
        $this->assertArrayHasKey('patient', $extraParams);
        $this->assertArrayHasKey('need_patient_banner', $extraParams);
        $this->assertArrayHasKey('scope', $extraParams);

        $this->assertEquals('standalone-patient-uuid', $extraParams['patient']);
        $this->assertTrue($extraParams['need_patient_banner']);
    }

    public function testContextForScopesWithExistingContextIntegration(): void
    {
        // Prepare existing context
        $existingContext = [
            'patient' => 'existing-patient-uuid',
            'encounter' => 'existing-encounter-uuid',
            'intent' => 'existing-intent'
        ];

        // Test with EHR launch scopes
        $ehrScopes = [$this->createMockScope(SmartLaunchController::CLIENT_APP_REQUIRED_LAUNCH_SCOPE)];

        $context = $this->contextBuilder->getContextForScopesWithExistingContext($existingContext, $ehrScopes);

        $this->assertIsArray($context);
        $this->assertArrayHasKey('patient', $context);
        $this->assertArrayHasKey('encounter', $context);
        $this->assertArrayHasKey('intent', $context);
        $this->assertArrayHasKey('need_patient_banner', $context);
        $this->assertArrayHasKey('smart_style_url', $context);

        $this->assertEquals('existing-patient-uuid', $context['patient']);
        $this->assertEquals('existing-encounter-uuid', $context['encounter']);
        $this->assertEquals('existing-intent', $context['intent']);
        $this->assertFalse($context['need_patient_banner']);

        // Test with standalone scopes
        $standaloneScopes = [$this->createMockScope(SmartLaunchController::CLIENT_APP_STANDALONE_LAUNCH_SCOPE)];

        $standaloneContext = $this->contextBuilder->getContextForScopesWithExistingContext($existingContext, $standaloneScopes);

        $this->assertIsArray($standaloneContext);
        $this->assertArrayHasKey('patient', $standaloneContext);
        $this->assertArrayHasKey('need_patient_banner', $standaloneContext);
        $this->assertArrayNotHasKey('encounter', $standaloneContext); // Should not include encounter for standalone

        $this->assertEquals('existing-patient-uuid', $standaloneContext['patient']);
        $this->assertTrue($standaloneContext['need_patient_banner']);
    }

    public function testInvalidLaunchContextThrowsException(): void
    {
        // Set invalid launch token in session
        $this->session->set('launch', 'invalid-launch-token');

        $scopes = [$this->createMockScope(SmartLaunchController::CLIENT_APP_REQUIRED_LAUNCH_SCOPE)];

        $this->expectException(OAuthServerException::class);
        $this->expectExceptionMessage('Invalid launch parameter');

        $this->contextBuilder->getContextForScopes($scopes);
    }

    public function testEmptyLaunchContextReturnsEmptyArray(): void
    {
        // Don't set any launch context in session
        $scopes = [$this->createMockScope(SmartLaunchController::CLIENT_APP_REQUIRED_LAUNCH_SCOPE)];

        $context = $this->contextBuilder->getContextForScopes($scopes);

        $this->assertIsArray($context);
        $this->assertEmpty($context);
    }

    public function testNoPatientInStandaloneLaunchReturnsEmptyContext(): void
    {
        // Don't set puuid in session
        $scopes = [$this->createMockScope(SmartLaunchController::CLIENT_APP_STANDALONE_LAUNCH_SCOPE)];

        $context = $this->contextBuilder->getContextForScopes($scopes);

        $this->assertIsArray($context);
        $this->assertEmpty($context);
    }

    public function testNonSMARTScopesReturnEmptyContext(): void
    {
        $scopes = [
            $this->createMockScope('openid'),
            $this->createMockScope('patient/*.read')
        ];

        $context = $this->contextBuilder->getContextForScopes($scopes);

        $this->assertIsArray($context);
        $this->assertEmpty($context);
    }

    public function testOfflineAccessScopeIntegration(): void
    {
        $scopes = [
            $this->createMockScope('openid'),
            $this->createMockScope(IdTokenSMARTResponse::SCOPE_OFFLINE_ACCESS)
        ];

        $accessToken = $this->createMockAccessTokenWithScopes($scopes);
        $this->setAccessToken($accessToken);

        // Mock response
        $response = (new Psr17Factory())->createResponse();

        // This should call the parent method due to offline_access scope
        $result = $this->idTokenResponse->generateHttpResponse($response);

        $this->assertInstanceOf(ResponseInterface::class, $result);
    }

    private function createMockScope(string $identifier): MockObject
    {
        $scope = $this->createMock(ScopeEntityInterface::class);
        $scope->method('getIdentifier')->willReturn($identifier);

        return $scope;
    }

    private function createMockAccessTokenWithScopes(array $scopes): MockObject
    {
        $accessToken = $this->createMock(AccessTokenEntityInterface::class);
        $client = $this->createMock(ClientEntityInterface::class);
        $client->method('getIdentifier')->willReturn('test-client-id');

        $accessToken->method('getClient')->willReturn($client);
        $accessToken->method('getUserIdentifier')->willReturn('test-user-id');
        $accessToken->method('getExpiryDateTime')->willReturn(new \DateTimeImmutable('+1 hour'));
        $accessToken->method('getScopes')->willReturn($scopes);
        $accessToken->method('__toString')->willReturn('mock_access_token');

        return $accessToken;
    }

    private function setAccessToken(AccessTokenEntityInterface $accessToken): void
    {
        // Use reflection to set the protected accessToken property
        $reflection = new \ReflectionClass($this->idTokenResponse);
        $property = $reflection->getProperty('accessToken');
        $property->setValue($this->idTokenResponse, $accessToken);
    }

    public function testCompleteEHRLaunchWorkflow(): void
    {
        // Simulate complete EHR launch workflow
        $launchToken = new SMARTLaunchToken();
        $launchToken->setPatient('workflow-patient-uuid');
        $launchToken->setEncounter('workflow-encounter-uuid');
        $launchToken->setIntent(SMARTLaunchToken::INTENT_PATIENT_DEMOGRAPHICS_DIALOG);

        $this->session->set('launch', $launchToken->serialize());
        $this->session->set('nonce', 'workflow-nonce');

        $scopes = [
            $this->createMockScope('openid'),
            $this->createMockScope(SmartLaunchController::CLIENT_APP_REQUIRED_LAUNCH_SCOPE),
            $this->createMockScope('patient/*.read'),
            $this->createMockScope('encounter/*.read')
        ];

        // Build context
        $context = $this->contextBuilder->getContextForScopes($scopes);

        // Set context in IdTokenSMARTResponse
        $this->idTokenResponse->setContextForNewTokens($context);

        // Create access token and set it
        $accessToken = $this->createMockAccessTokenWithScopes($scopes);
        $this->setAccessToken($accessToken);

        // Mock user entity for ID token creation
        $userEntity = $this->createMock(UserEntityInterface::class);
        $userEntity->method('getIdentifier')->willReturn('workflow-user-id');

        $claimSetEntity = $this->createMock(ClaimSetInterface::class);
        $claimSetEntity->method('getClaims')->willReturn(['sub' => 'workflow-user-id']);

        $combinedEntity = new class implements UserEntityInterface, ClaimSetInterface {
            public function getIdentifier()
            {
                return 'workflow-user-id'; }
            public function getClaims()
            {
                return ['sub' => 'workflow-user-id']; }
        };

        $this->identityProvider->method('getUserEntityByIdentifier')
            ->willReturn($combinedEntity);

        $this->claimExtractor->method('extract')
            ->willReturn(['sub' => 'workflow-user-id']);

        // Test the complete flow through getExtraParams
        $reflection = new \ReflectionClass($this->idTokenResponse);
        $method = $reflection->getMethod('getExtraParams');

        $extraParams = $method->invoke($this->idTokenResponse, $accessToken);

        // Verify all expected parameters are present
        $this->assertIsArray($extraParams);
        $this->assertArrayHasKey('patient', $extraParams);
        $this->assertArrayHasKey('encounter', $extraParams);
        $this->assertArrayHasKey('intent', $extraParams);
        $this->assertArrayHasKey('need_patient_banner', $extraParams);
        $this->assertArrayHasKey('smart_style_url', $extraParams);
        $this->assertArrayHasKey('scope', $extraParams);
        $this->assertArrayHasKey('id_token', $extraParams); // From parent class

        // Verify values
        $this->assertEquals('workflow-patient-uuid', $extraParams['patient']);
        $this->assertEquals('workflow-encounter-uuid', $extraParams['encounter']);
        $this->assertEquals(SMARTLaunchToken::INTENT_PATIENT_DEMOGRAPHICS_DIALOG, $extraParams['intent']);
        $this->assertFalse($extraParams['need_patient_banner']);
        $this->assertStringContainsString('smart-style', $extraParams['smart_style_url']);

        // Verify scope string formatting
        $expectedScopes = 'openid ' . SmartLaunchController::CLIENT_APP_REQUIRED_LAUNCH_SCOPE . ' patient/*.read encounter/*.read';
        $this->assertEquals($expectedScopes, $extraParams['scope']);
    }

    public function testCompleteStandaloneLaunchWorkflow(): void
    {
        // Simulate complete standalone launch workflow
        $this->session->set('puuid', 'standalone-workflow-patient');
        $this->session->set('nonce', 'standalone-workflow-nonce');

        $scopes = [
            $this->createMockScope('openid'),
            $this->createMockScope(SmartLaunchController::CLIENT_APP_STANDALONE_LAUNCH_SCOPE),
            $this->createMockScope('patient/*.read')
        ];

        // Build context
        $context = $this->contextBuilder->getContextForScopes($scopes);

        // Set context in IdTokenSMARTResponse
        $this->idTokenResponse->setContextForNewTokens($context);

        // Create access token and set it
        $accessToken = $this->createMockAccessTokenWithScopes($scopes);
        $this->setAccessToken($accessToken);

        // Mock user entity for ID token creation
        $combinedEntity = new class implements UserEntityInterface, ClaimSetInterface {
            public function getIdentifier()
            {
                return 'standalone-user-id'; }
            public function getClaims()
            {
                return ['sub' => 'standalone-user-id']; }
        };

        $this->identityProvider->method('getUserEntityByIdentifier')
            ->willReturn($combinedEntity);

        $this->claimExtractor->method('extract')
            ->willReturn(['sub' => 'standalone-user-id']);

        // Test the complete flow
        $reflection = new \ReflectionClass($this->idTokenResponse);
        $method = $reflection->getMethod('getExtraParams');

        $extraParams = $method->invoke($this->idTokenResponse, $accessToken);

        // Verify standalone-specific parameters
        $this->assertIsArray($extraParams);
        $this->assertArrayHasKey('patient', $extraParams);
        $this->assertArrayHasKey('need_patient_banner', $extraParams);
        $this->assertArrayHasKey('smart_style_url', $extraParams);
        $this->assertArrayHasKey('scope', $extraParams);
        $this->assertArrayHasKey('id_token', $extraParams);

        // Verify values specific to standalone launch
        $this->assertEquals('standalone-workflow-patient', $extraParams['patient']);
        $this->assertTrue($extraParams['need_patient_banner']); // Should be true for standalone
        $this->assertStringContainsString('smart-style', $extraParams['smart_style_url']);

        // Should not include encounter or intent for standalone
        $this->assertArrayNotHasKey('encounter', $extraParams);
        $this->assertArrayNotHasKey('intent', $extraParams);
    }

    public function testRefreshTokenContextPreservation(): void
    {
        // Simulate refresh token scenario where we need to preserve existing context
        $originalContext = [
            'patient' => 'refresh-patient-uuid',
            'encounter' => 'refresh-encounter-uuid',
            'intent' => 'refresh-intent',
            'need_patient_banner' => false,
            'smart_style_url' => 'https://example.com/styles.json'
        ];

        $scopes = [
            $this->createMockScope('openid'),
            $this->createMockScope(SmartLaunchController::CLIENT_APP_REQUIRED_LAUNCH_SCOPE),
            $this->createMockScope('patient/*.read')
        ];

        // Test context preservation with existing context
        $preservedContext = $this->contextBuilder->getContextForScopesWithExistingContext($originalContext, $scopes);

        $this->assertEquals($originalContext['patient'], $preservedContext['patient']);
        $this->assertEquals($originalContext['encounter'], $preservedContext['encounter']);
        $this->assertEquals($originalContext['intent'], $preservedContext['intent']);
        $this->assertFalse($preservedContext['need_patient_banner']); // EHR launch default

        // Set the preserved context in the response
        $this->idTokenResponse->setContextForNewTokens($preservedContext);

        $accessToken = $this->createMockAccessTokenWithScopes($scopes);
        $this->setAccessToken($accessToken);

        $reflection = new \ReflectionClass($this->idTokenResponse);
        $method = $reflection->getMethod('getExtraParams');

        $extraParams = $method->invoke($this->idTokenResponse, $accessToken);

        // Verify context is preserved in refresh scenario
        $this->assertEquals('refresh-patient-uuid', $extraParams['patient']);
        $this->assertEquals('refresh-encounter-uuid', $extraParams['encounter']);
        $this->assertEquals('refresh-intent', $extraParams['intent']);
    }

    public function testScopeIdentificationMethods(): void
    {
        $ehrScopes = [$this->createMockScope(SmartLaunchController::CLIENT_APP_REQUIRED_LAUNCH_SCOPE)];
        $standaloneScopes = [$this->createMockScope(SmartLaunchController::CLIENT_APP_STANDALONE_LAUNCH_SCOPE)];
        $regularScopes = [$this->createMockScope('openid'), $this->createMockScope('patient/*.read')];

        // Test EHR launch identification
        $this->assertTrue($this->contextBuilder->isEHRLaunchRequest($ehrScopes));
        $this->assertFalse($this->contextBuilder->isEHRLaunchRequest($standaloneScopes));
        $this->assertFalse($this->contextBuilder->isEHRLaunchRequest($regularScopes));

        // Test standalone launch identification
        $this->assertTrue($this->contextBuilder->isStandaloneLaunchPatientRequest($standaloneScopes));
        $this->assertFalse($this->contextBuilder->isStandaloneLaunchPatientRequest($ehrScopes));
        $this->assertFalse($this->contextBuilder->isStandaloneLaunchPatientRequest($regularScopes));
    }

    public function testLoggingIntegration(): void
    {
        // Test that logging methods are called appropriately
        $launchToken = new SMARTLaunchToken();
        $launchToken->setPatient('logging-test-patient');

        $this->session->set('launch', $launchToken->serialize());

        // Expect debug logging for launch context
        $this->logger->expects($this->atLeastOnce())
            ->method('debug')
            ->with(
                $this->stringContains('SMARTSessionTokenContextBuilder'),
                $this->isType('array')
            );

        $scopes = [$this->createMockScope(SmartLaunchController::CLIENT_APP_REQUIRED_LAUNCH_SCOPE)];
        $context = $this->contextBuilder->getContextForScopes($scopes);

        $this->assertIsArray($context);
    }

    public function testErrorHandlingInLaunchContextExtraction(): void
    {
        // Test error handling when launch token deserialization fails
        $this->session->set('launch', 'corrupted-token-data');

        $scopes = [$this->createMockScope(SmartLaunchController::CLIENT_APP_REQUIRED_LAUNCH_SCOPE)];

        // Expect error logging
        $this->logger->expects($this->once())
            ->method('error')
            ->with(
                $this->stringContains('Failed to decode launch context parameter'),
                $this->arrayHasKey('error')
            );

        $this->expectException(OAuthServerException::class);
        $this->expectExceptionMessage('Invalid launch parameter');

        $this->contextBuilder->getContextForScopes($scopes);
    }

    public function testSmartStyleUrlGeneration(): void
    {
        // Test that SMART style URL is properly generated
        $this->session->set('puuid', 'style-test-patient');

        $scopes = [$this->createMockScope(SmartLaunchController::CLIENT_APP_STANDALONE_LAUNCH_SCOPE)];
        $context = $this->contextBuilder->getContextForScopes($scopes);

        $this->assertArrayHasKey('smart_style_url', $context);
        $this->assertStringContainsString('https://example.com', $context['smart_style_url']);
        $this->assertStringContainsString('/openemr/oauth2/default', $context['smart_style_url']);
        $this->assertStringContainsString('smart-style', $context['smart_style_url']);
    }
}
