<?php

namespace OpenEMR\Tests\Certification\HIT1\US_Core_311;

use GuzzleHttp\Client;
use OpenEMR\Tests\Api\ApiTestClient;
use PHPUnit\Framework\TestCase;

class InfernoSinglePatientAPITest extends TestCase
{
    // Alice Jones (96506861-511f-4f6d-bc97-b65a78cf1995),
    // Jeremy Bates (96891ab2-01ad-49f9-9958-cdad71bd33c1),
    // Happy Child(968944d0-180a-48ac-8049-636ae8ac6127),
    // and John Appleseed (969f72c3-0256-488e-b25b-8fff3af18b1b)
    // are the patients used in the Inferno Single Patient API tests.
    const PATIENT_IDS = '96506861-511f-4f6d-bc97-b65a78cf1995,96891ab2-01ad-49f9-9958-cdad71bd33c1,968944d0-180a-48ac-8049-636ae8ac6127';
    const PATIENT_ID_PRIMARY = '96506861-511f-4f6d-bc97-b65a78cf1995';

    // we break them apart due to some wierdisness in the Inferno Single Patient API tests
    const ADDITIONAL_PATIENT_IDS = '96891ab2-01ad-49f9-9958-cdad71bd33c1,968944d0-180a-48ac-8049-636ae8ac6127';

    const TEST_SUITE_US_CORE_V311 = 'us_core_v311'; // this is the test suite id for the US Core v3.11 tests
    const TEST_SUITE_G10_CERTIFICATION = 'g10_certification'; // this is the test suite id for the US Core v3.11 certification tests

    const TEST_SUITE = self::TEST_SUITE_US_CORE_V311;
    const DEFAULT_OPENEMR_BASE_URL_API = 'http://openemr';
    const DEFAULT_INFERNO_BASE_URL = 'http://nginx';
    const DEFAULT_TEST_GROUP_ID = 'us_core_v311-us_core_v311_fhir_api';
    private static ApiTestClient $testClient;
    private static string $baseUrl;

    private static Client $infernoClient;

    private static ?string $sessionId;

    const TIMEOUT = 60; // seconds

    public static function setUpBeforeClass(): void
    {
        parent::setUpBeforeClass(); // TODO: Change the autogenerated stub
        $baseUrl = getenv("OPENEMR_BASE_URL_API", true) ?: self::DEFAULT_OPENEMR_BASE_URL_API;
        self::$testClient = new ApiTestClient($baseUrl, false);
        self::$baseUrl = $baseUrl;
        // for now this uses the admin user to authenticate
        // TODO: @adunsulag need to implement this using a test practitioner user so we can test the inferno single patient API from a regular provider
        self::$testClient->setAuthToken(ApiTestClient::OPENEMR_AUTH_ENDPOINT);
        $accessToken = self::$testClient->getAccessToken();
        $infernoUrl = getenv("INFERNO_BASE_URL", true) ?: self::DEFAULT_INFERNO_BASE_URL;
        self::$infernoClient = new Client(['timeout' => self::TIMEOUT, 'base_uri' => $infernoUrl . '/api/']);
        $response = self::$infernoClient->post('test_sessions?test_suite_id=us_core_v311');
        $jsonResponse = json_decode($response->getBody(), true);
        self::$sessionId = $jsonResponse['id'] ?? null;
        if (!self::$sessionId) {
            throw new \Exception("Failed to create test session for Inferno Single Patient API tests");
        }
    }

//
//    public static function tearDownAfterClass() : void
//    {
//        parent::tearDownAfterClass(); // TODO: Change the autogenerated stub
//        self::$testClient->cleanupClient();
//        self::$sessionId = null;
//    }

    public function testCapabilityStatement(): void
    {
        $response = $this->getTestGroupResponse($this->getTestSuitePrefix() . 'capability_statement');
        $this->assertResultsPassed(
            $response['results'],
            "Capability Statement test failed",
            [
            // we skip the standalone auth TLS test for now as the unit test environment does not support TLS
            'us_core_v311-us_core_v311_fhir_api-us_core_v311_capability_statement-standalone_auth_tls'
            // we skip the overall group failure test as the sub test failing triggers the group failure
            ,'us_core_v311-us_core_v311_fhir_api-us_core_v311_capability_statement'
            ]
        );
    }

    public function testPatient(): void
    {
        $response = $this->getTestGroupResponse($this->getTestSuitePrefix() . 'patient', 'smart_auth_info');
        $this->assertResultsPassed($response['results'], "Patient Resource test failed");
    }

    public function testAllergyIntolerance(): void
    {
        $response = $this->getTestGroupResponse($this->getTestSuitePrefix() . 'allergy_intolerance', 'smart_auth_info');
//        $response = $this->getTestGroupResponse('g10_certification-g10_single_patient_api-us_core_v311_allergy_intolerance', 'smart_auth_info');
        $this->assertResultsPassed($response['results'], "AllergyIntolerance Resource test failed");
    }

    public function testCarePlan(): void
    {
        $response = $this->getTestGroupResponse($this->getTestSuitePrefix() . 'care_plan', 'smart_auth_info');
        $this->assertResultsPassed($response['results'], "CarePlan Resource test failed");
    }

    public function testCondition(): void
    {
        $response = $this->getTestGroupResponse($this->getTestSuitePrefix() . 'condition', 'smart_auth_info');
        $this->assertResultsPassed($response['results'], "Condition Resource test failed");
    }

    public function testDevice(): void
    {
        $response = $this->getTestGroupResponse($this->getTestSuitePrefix() . 'device', 'smart_auth_info');
        $this->assertResultsPassed($response['results'], "Device Resource test failed");
    }
    public function testDiagnosticReportNote(): void
    {
        $response = $this->getTestGroupResponse($this->getTestSuitePrefix() . 'diagnostic_report_note', 'smart_auth_info');
        $this->assertResultsPassed($response['results'], "DiagnosticReport and Note exchange test failed");
    }

    public function testDiagnosticReportLab(): void
    {
        $response = $this->getTestGroupResponse($this->getTestSuitePrefix() . 'diagnostic_report_lab', 'smart_auth_info');
        $this->assertResultsPassed($response['results'], "DiagnosticReport Laboratory Resource test failed");
    }

    public function testDocumentReference(): void
    {
        $response = $this->getTestGroupResponse($this->getTestSuitePrefix() . 'document_reference', 'smart_auth_info');
        $this->assertResultsPassed($response['results'], "Document Reference Resource test failed");
    }

    public function testGoal(): void
    {
        $response = $this->getTestGroupResponse($this->getTestSuitePrefix() . 'goal', 'smart_auth_info');
        $this->assertResultsPassed($response['results'], "Goal Resource test failed");
    }

    public function testImmunization(): void
    {
        $response = $this->getTestGroupResponse($this->getTestSuitePrefix() . 'immunization', 'smart_auth_info');
        $this->assertResultsPassed($response['results'], "Immunization Resource test failed");
    }

    public function testMedicationRequest(): void
    {
        $response = $this->getTestGroupResponse($this->getTestSuitePrefix() . 'medication_request', 'smart_auth_info');
        $this->assertResultsPassed($response['results'], "MedicationRequest Resource test failed");
    }

    public function testSmokingStatus(): void
    {
        $response = $this->getTestGroupResponse($this->getTestSuitePrefix() . 'smokingstatus', 'smart_auth_info');
        $this->assertResultsPassed($response['results'], "Us Core Smoking Status Observation Resource test failed");
    }

    public function testPediatricWeightForHeight(): void
    {
        $response = $this->getTestGroupResponse($this->getTestSuitePrefix() . 'pediatric_weight_for_height', 'smart_auth_info');
        $this->assertResultsPassed($response['results'], "Us Core Pediatric Weight for Height Observation Resource test failed");
    }

    public function testObservationLab(): void
    {
        $response = $this->getTestGroupResponse($this->getTestSuitePrefix() . 'observation_lab', 'smart_auth_info');
        $this->assertResultsPassed($response['results'], "Observation Laboratory Resource test failed");
    }

    public function testPediatricBmiForAge(): void
    {
        $response = $this->getTestGroupResponse($this->getTestSuitePrefix() . 'pediatric_bmi_for_age', 'smart_auth_info');
        $this->assertResultsPassed($response['results'], "Us Core Pediatric BMI for Age Observation Resource test failed");
    }

    public function testPulseOximetry(): void
    {
        $response = $this->getTestGroupResponse($this->getTestSuitePrefix() . 'pulse_oximetry', 'smart_auth_info');
        $this->assertResultsPassed($response['results'], "Us Core Pulse Oximetry Observation Resource test failed");
    }

    public function testHeadCircumference(): void
    {
        $response = $this->getTestGroupResponse($this->getTestSuitePrefix() . 'head_circumference', 'smart_auth_info');
        $this->assertResultsPassed($response['results'], "Us Core Head Circumference Observation Resource test failed");
    }

    public function testBodyHeight(): void
    {
        $response = $this->getTestGroupResponse($this->getTestSuitePrefix() . 'body_height', 'smart_auth_info');
        $this->assertResultsPassed($response['results'], "Us Core Body Height Observation Resource test failed");
    }

    public function testBodyTemp(): void
    {
        $response = $this->getTestGroupResponse($this->getTestSuitePrefix() . 'bodytemp', 'smart_auth_info');
        $this->assertResultsPassed($response['results'], "Us Core Body Temperature Observation Resource test failed");
    }

    public function testBloodPressure(): void
    {
        $response = $this->getTestGroupResponse($this->getTestSuitePrefix() . 'bp', 'smart_auth_info');
        $this->assertResultsPassed($response['results'], "Us Core Blood Pressure Observation Resource test failed");
    }

    public function testBodyWeight(): void
    {
        $response = $this->getTestGroupResponse($this->getTestSuitePrefix() . 'body_weight', 'smart_auth_info');
        $this->assertResultsPassed($response['results'], "Us Core Body Weight Observation Resource test failed");
    }

    public function testHeartRate(): void
    {
        $response = $this->getTestGroupResponse($this->getTestSuitePrefix() . 'heartrate', 'smart_auth_info');
        $this->assertResultsPassed($response['results'], "Us Core Heart rate Observation Resource test failed");
    }

    public function testRespRate(): void
    {
        $response = $this->getTestGroupResponse($this->getTestSuitePrefix() . 'resprate', 'smart_auth_info');
        $this->assertResultsPassed($response['results'], "Us Core Respitatory Rate Observation Resource test failed");
    }

    public function testProcedure(): void
    {
        $response = $this->getTestGroupResponse($this->getTestSuitePrefix() . 'procedure', 'smart_auth_info');
        $this->assertResultsPassed($response['results'], "Us Core Procedure Resource test failed");
    }

    public function testEncounter(): void
    {
        $response = $this->getTestGroupResponse($this->getTestSuitePrefix() . 'encounter', 'smart_auth_info');
        $this->assertResultsPassed($response['results'], "Us Core Encounter Resource test failed");
    }

    public function testOrganization(): void
    {
        $response = $this->getTestGroupResponse($this->getTestSuitePrefix() . 'encounter', 'smart_auth_info');
        $this->assertResultsPassed($response['results'], "Us Core Organization Resource test failed");
    }

    public function testPractitioner(): void
    {
        $response = $this->getTestGroupResponse($this->getTestSuitePrefix() . 'encounter', 'smart_auth_info');
        $this->assertResultsPassed($response['results'], "Us Core Practitioner Resource test failed");
    }

    public function testClinicalNotesGuidance(): void
    {
        $response = $this->getTestGroupResponse($this->getTestSuitePrefix() . 'clinical_notes_guidance', 'smart_auth_info');
        $this->assertResultsPassed($response['results'], "Us Core Clinical Notes Guidance test failed");
    }

    public function testDataAbsentReason(): void
    {
        $response = $this->getTestGroupResponse($this->getTestSuitePrefix() . 'data_absent_reason', 'smart_auth_info');
        $this->assertResultsPassed($response['results'], "Us Core Data Absent Reason test failed");
    }

    // we do provenance last as it is the most complex test and can take a while to run
    public function testProvenance(): void
    {
        $response = $this->getTestGroupResponse($this->getTestSuitePrefix() . 'provenance', 'smart_auth_info');
        $this->assertResultsPassed($response['results'], "Us Core Provenance Resource test failed");
    }


    protected function getTestSuitePrefix()
    {
        if (self::TEST_SUITE === self::TEST_SUITE_US_CORE_V311) {
            return 'us_core_v311-us_core_v311_fhir_api-us_core_v311_';
        } elseif (self::TEST_SUITE === self::TEST_SUITE_G10_CERTIFICATION) {
            return 'g10_certification-g10_single_patient_api-us_core_v311_';
        } else {
            throw new \Exception("Unknown test suite: " . self::TEST_SUITE);
        }
    }
    protected function assertResultsPassed(array $results, string $assertMessage, array $testIdsToSkipFailures = []): void
    {
        foreach ($results as $result) {
            $failMessage = '';
            if ($result['result'] === 'skip') {
                continue; // skip this test if it's skipped
            }
            if ($result['result'] !== 'pass') {
//                var_dump($result);
                if (!empty($result['test_id'])) {
                    if (in_array($result['test_id'], $testIdsToSkipFailures)) {
                        continue; // skip this test if it's in the skip list
                    } else {
                        $failMessage = $assertMessage . " for test " . $result['test_id'] . ' ' . $result['result_message'];
                    }
                } elseif (!empty($result['test_group_id'])) {
                    if (in_array($result['test_group_id'], $testIdsToSkipFailures)) {
                        continue; // skip this test if it's in the skip list
                    } else {
                        $failMessage = $assertMessage . " for test group " . $result['test_group_id'];
                    }
                } else {
                    $failMessage = $assertMessage . ' ' . $result['result_message'];
                }
                if (!empty($result['requests'])) {
                    foreach ($result['requests'] as $request) {
                        if ($request['result_id'] == $result['id']) {
                            $failMessage .= "\nRequest: " . $request['verb'] . " " . $request['url'] . " status " . $request['status'];
                        }
                    }
                }
                if (!empty($result['messages'])) {
                    foreach ($result['messages'] as $message) {
                        $failMessage .= "\nMessage(type=" . $message['type'] . "): " . $message['message'];
                    }
                }
            }
            $this->assertEquals("pass", $result['result'], $failMessage);
        }
    }

    protected function getTestInputs(string $credentialsKeyName, string $accessToken): array
    {
        if (self::TEST_SUITE == self::TEST_SUITE_G10_CERTIFICATION) {
            return [
                ['name' => 'url', 'value' => self::$baseUrl . '/apis/default/fhir'],
                ['name' => 'patient_id', 'value' => self::PATIENT_ID_PRIMARY],
                ['name' => 'patient_ids', 'value' => self::PATIENT_IDS],
                ['name' => 'additional_patient_ids', 'value' => self::ADDITIONAL_PATIENT_IDS],
                ['name' => $credentialsKeyName, 'value' => ['access_token' => $accessToken]]
            ];
        } else {
            return [
                ['name' => 'url', 'value' => self::$baseUrl . '/apis/default/fhir'],
                ['name' => 'patient_ids', 'value' => self::PATIENT_IDS],
                ['name' => $credentialsKeyName, 'value' => ['access_token' => $accessToken]]
            ];
        }
    }


    protected function getTestGroupResponse(string $testGroupId, $credentialsKeyName = 'smart_credentials'): array
    {
        $accessToken = self::$testClient->getAccessToken();
        $testRunData = [
            'test_session_id' => self::$sessionId,
            'test_group_id' => $testGroupId,
            'inputs' => $this->getTestInputs($credentialsKeyName, $accessToken)
        ];
        $testRunResponse = self::$infernoClient->post('test_runs', [
            'json' => $testRunData
        ]);
        // we grab the response.id JSON value from the result and store it in $testRunId
        $this->assertEquals(200, $testRunResponse->getStatusCode(), "Failed to create test run for " . $testGroupId);
        $testRunJsonResponse = json_decode($testRunResponse->getBody(), true);
        $testRunId = $testRunJsonResponse['id'] ?? null;
        $this->assertNotNull($testRunId, "Test run ID is null");

        // we then need to do a polling request every second to the inferno /test_runs/$testRunId?include_results=false
        // until the status is 'done'
        // if after 60 seconds the status is not 'done', we fail the test
        $maxRetries = self::TIMEOUT * 2; // 60 seconds
        $retryCount = 0;
        $status = '';
        while ($retryCount < $maxRetries) {
            $testRunStatusResponse = self::$infernoClient->get("test_runs/{$testRunId}?include_results=false");
            $this->assertEquals(200, $testRunStatusResponse->getStatusCode(), "Failed to get test run status for " . $testGroupId);
            $testRunStatusJson = json_decode($testRunStatusResponse->getBody(), true);
            $status = $testRunStatusJson['status'] ?? '';
            if ($status === 'done') {
                break;
            }
            usleep(500000); // sleep for 500 milliseconds
            $retryCount++;
        }

        // once the status is 'done', we can make a final request to inferno /test_runs/$testRunId?include_results=true
        // and we can then verify the results against the expected results for the Inferno Single Patient API tests.
        $this->assertEquals('done', $status, "Test run did not complete in time");
        $finalTestRunResponse = self::$infernoClient->get("test_runs/{$testRunId}?include_results=true");
        $this->assertEquals(200, $finalTestRunResponse->getStatusCode(), "Failed to get final test run results for " . $testGroupId);
        $finalTestRunJson = json_decode($finalTestRunResponse->getBody(), true);
        $this->assertNotEmpty($finalTestRunJson['results'], "Test run results are empty for " . $testGroupId);
        return $finalTestRunJson;
    }
}
