<?php

/**
 * GeoTelemetry class
 *
 * Over 80% of GeoTelemetry class code was generated by ChatGPT code copilot.
 * The rest was modified to fit the OpenEMR coding style.
 * Claude.ai was used for code review and optimization suggestions.
 *
 * Lightweight geolocation and anonymization utility
 * - Uses public IP lookup APIs
 * - Composer-free
 * - Fallback support and IP anonymization
 * - Caching to prevent rate limiting
 *
 * @package        OpenEMR
 * @link           https://www.open-emr.org
 * @link           https://opencoreemr.com
 * @author         Jerry Padgett <sjpadgett@gmail.com>
 * @author         Michael A. Smith <michael@opencoreemr.com>
 * @copyright      Copyright (c) 2025 - 2026 <sjpadgett@gmail.com>
 * @copyright      Copyright (c) 2026 OpenCoreEMR Inc.
 * @license        https://github.com/openemr/openemr/blob/master/LICENSE GNU General Public License 3
 */

namespace OpenEMR\Telemetry;

use OpenEMR\Common\Utils\ValidationUtils;
use OpenEMR\Services\Globals\GlobalSetting;

class GeoTelemetry implements GeoTelemetryInterface
{
    private const GET_IP_URL = 'https://api.ipify.org';
    private const CACHE_FILENAME = 'openemr_geo_cache.json';
    private const CACHE_DURATION = 86400; // 24 hours in seconds

    /**
     * Get the cache file path from OpenEMR temp directory
     */
    private function getCacheFilePath(): string
    {
        return sys_get_temp_dir() . '/cache/' . self::CACHE_FILENAME;
    }

    /**
     * Anonymize IP using SHA-256 hashing
     */
    public function anonymizeIp(string $ip): string
    {
        return hash('sha256', $ip);
    }

    /**
     * Get geolocation data from IP using a lightweight external API
     * Now with caching to prevent rate limiting
     */
    public function getGeoData(string $ip): array
    {
        // Check cache first
        $cachedData = $this->getCachedGeoData($ip);
        if ($cachedData !== null) {
            return $cachedData;
        }

        // Try multiple providers in sequence
        $providers = [
            fn() => $this->tryIpApi($ip),
            fn() => $this->tryGeoPlugin($ip),
            fn() => $this->tryIpApiCom($ip),
        ];

        foreach ($providers as $provider) {
            $data = $provider();
            if (!isset($data['error'])) {
                // Cache successful result
                $this->cacheGeoData($ip, $data);
                return $data;
            }
        }

        // All providers failed
        return ['error' => 'All IP lookup providers failed'];
    }

    /**
     * Get geolocation of the current server (public-facing IP)
     */
    public function getServerGeoData(): array
    {
        $ip = trim($this->fetchText(self::GET_IP_URL));
        if (ValidationUtils::isValidIpAddress($ip)) {
            return $this->getGeoData($ip);
        }
        return ['error' => 'Unable to determine server IP'];
    }

    /**
     * Try ipapi.co provider
     */
    private function tryIpApi(string $ip): array
    {
        $data = $this->fetchJson("https://ipapi.co/{$ip}/json/");
        if (isset($data['country_name']) && !isset($data['error'])) {
            return [
                'country' => $data['country_name'] ?? null,
                'region' => $data['region'] ?? null,
                'city' => $data['city'] ?? null,
                'latitude' => $data['latitude'] ?? null,
                'longitude' => $data['longitude'] ?? null,
            ];
        }
        return ['error' => $data['reason'] ?? 'ipapi.co lookup failed'];
    }

    /**
     * Try geoplugin.net provider
     */
    private function tryGeoPlugin(string $ip): array
    {
        $data = $this->fetchJson("http://www.geoplugin.net/json.gp?ip={$ip}");
        if (isset($data['geoplugin_countryName']) && $data['geoplugin_countryName'] !== null) {
            return [
                'country' => $data['geoplugin_countryName'] ?? null,
                'region' => $data['geoplugin_region'] ?? null,
                'city' => $data['geoplugin_city'] ?? null,
                'latitude' => $data['geoplugin_latitude'] ?? null,
                'longitude' => $data['geoplugin_longitude'] ?? null,
            ];
        }
        return ['error' => 'geoplugin.net lookup failed'];
    }

    /**
     * Try ip-api.com provider (free, no auth required, 45 req/min)
     */
    private function tryIpApiCom(string $ip): array
    {
        $data = $this->fetchJson("http://ip-api.com/json/{$ip}");
        if (isset($data['status']) && $data['status'] === 'success') {
            return [
                'country' => $data['country'] ?? null,
                'region' => $data['regionName'] ?? null,
                'city' => $data['city'] ?? null,
                'latitude' => $data['lat'] ?? null,
                'longitude' => $data['lon'] ?? null,
            ];
        }
        return ['error' => $data['message'] ?? 'ip-api.com lookup failed'];
    }

    /**
     * Get cached geolocation data for an IP
     */
    private function getCachedGeoData(string $ip): ?array
    {
        $cacheFile = $this->getCacheFilePath();

        if (!file_exists($cacheFile)) {
            return null;
        }

        $cacheContent = @file_get_contents($cacheFile);
        if ($cacheContent === false) {
            return null;
        }

        $cache = json_decode($cacheContent, true);
        if (!is_array($cache)) {
            return null;
        }

        $cacheKey = $this->anonymizeIp($ip);
        if (!isset($cache[$cacheKey])) {
            return null;
        }

        $entry = $cache[$cacheKey];
        // Check if cache is still valid
        if (isset($entry['timestamp']) && (time() - $entry['timestamp']) < self::CACHE_DURATION) {
            return $entry['data'];
        }

        return null;
    }

    /**
     * Cache geolocation data for an IP
     */
    private function cacheGeoData(string $ip, array $data): void
    {
        $cacheFile = $this->getCacheFilePath();
        $cache = [];

        if (file_exists($cacheFile)) {
            $cacheContent = @file_get_contents($cacheFile);
            if ($cacheContent !== false) {
                $cache = json_decode($cacheContent, true) ?: [];
            }
        }

        // Clean up expired entries
        $now = time();
        foreach ($cache as $key => $entry) {
            if (isset($entry['timestamp']) && ($now - $entry['timestamp']) >= self::CACHE_DURATION) {
                unset($cache[$key]);
            }
        }

        // Add new entry
        $cacheKey = $this->anonymizeIp($ip);
        $cache[$cacheKey] = [
            'timestamp' => $now,
            'data' => $data
        ];

        // Write cache file
        @file_put_contents($cacheFile, json_encode($cache), LOCK_EX);
    }

    /**
     * Get raw text data from a given URL
     *
     * @param string $url
     * @return string
     */
    protected function fetchText(string $url): string
    {
        $ctx = stream_context_create([
            'http' => [
                'timeout' => 5,
                'user_agent' => 'OpenEMR-Telemetry/1.0',
                'ignore_errors' => true
            ]
        ]);

        $result = @$this->fileGetContents($url, false, $ctx);

        // Check if we got an HTTP error
        /** @phpstan-ignore-next-line */
        if ($result === false && isset($http_response_header)) {
            // Parse status code from headers
            /** @phpstan-ignore-next-line */
            if (preg_match('/HTTP\/\d\.\d\s+(\d+)/', (string)$http_response_header[0], $matches)) {
                $statusCode = (int)$matches[1];
                if ($statusCode >= 400) {
                    error_log("GeoTelemetry: HTTP {$statusCode} error fetching {$url}");
                }
            }
        }

        return $result ?: '';
    }

    /**
     * Get geolocation data from a given URL
     *
     * @param string $url
     * @return array
     */
    protected function fetchJson(string $url): array
    {
        $json = $this->fetchText($url);
        if (empty($json)) {
            return ['error' => 'Empty response'];
        }

        $data = json_decode($json, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            error_log("GeoTelemetry: JSON decode error: " . json_last_error_msg());
            return ['error' => 'Invalid JSON response'];
        }

        return $data ?: ['error' => 'Empty decoded data'];
    }

    /**
     * Wrapper so we can easily mock file_get_contents
     *
     * @codeCoverageIgnore
     */
    protected function fileGetContents(string $url, $use_include_path = false, $context = null, $offset = 0, $maxlen = null)
    {
        return file_get_contents($url, $use_include_path, $context, $offset, $maxlen);
    }
}
