<?php

/**
 * ObservationService handles all database operations for the observation form
 * AI Generated Note: Refactored to use new database schema from table.sql and support Design 1 implementation
 *
 * @package openemr
 * @link      http://www.open-emr.org
 * @author    Jacob T Paul <jacob@zhservices.com>
 * @author    Vinish K <vinish@zhservices.com>
 * @author    Brady Miller <brady.g.miller@gmail.com>
 * @author    Claude.AI on August 27th 2025
 * @author    Stephen Nielson <snielson@discoverandchange.com>
 * @copyright Copyright (c) 2015 Z&H Consultancy Services Private Limited <sam@zhservices.com>
 * @copyright Copyright (c) 2017-2019 Brady Miller <brady.g.miller@gmail.com>
 * @copyright Public Domain for the pieces that were generated by Claude.AI (refactor from interface/forms/observation) view.php,new.php,save.php
 * @license   https://github.com/openemr/openemr/blob/master/LICENSE GNU General Public License 3
 */

namespace OpenEMR\Services;

use OpenEMR\Common\Logging\SystemLoggerAwareTrait;
use OpenEMR\Services\Search\TokenSearchField;
use RuntimeException;
use OpenEMR\Common\Database\QueryUtils;
use OpenEMR\Common\Forms\ReasonStatusCodes;
use OpenEMR\Common\Uuid\UuidRegistry;
use OpenEMR\Services\Search\FhirSearchWhereClauseBuilder;
use OpenEMR\Services\Search\ISearchField;
use OpenEMR\Services\Utils\DateFormatterUtils;
use OpenEMR\Validators\ProcessingResult;
use Exception;

class ObservationService extends BaseService
{
    use SystemLoggerAwareTrait;

    const TABLE_NAME = 'form_observation';
    const FORM_NAME = "Observation Form";
    const FORM_DIR = "observation";

    private UuidRegistry $uuidRegistry;

    private ListService $listService;

    private FormService $formService;

    private CodeTypesService $codeTypesService;

    private array $typesById;

    private array $statiiById;

    /**
     * @var string[]
     */
    private array $reasonCodes;

    public function __construct()
    {
        parent::__construct(self::TABLE_NAME);
    }

    public function getFormService(): FormService
    {
        if (!isset($this->formService)) {
            $this->formService = new FormService();
        }
        return $this->formService;
    }

    public function setFormService(FormService $formService): void
    {
        $this->formService = $formService;
    }

    public function getListService(): ListService
    {
        if (!isset($this->listService)) {
            $this->listService = new ListService();
        }
        return $this->listService;
    }

    public function setListService(ListService $listService): void
    {
        $this->listService = $listService;
    }

    public function getCodeTypesService(): CodeTypesService
    {
        if (!isset($this->codeTypesService)) {
            $this->codeTypesService = new CodeTypesService();
        }
        return $this->codeTypesService;
    }
    public function setCodeTypesService(CodeTypesService $codeTypesService): void
    {
        $this->codeTypesService = $codeTypesService;
    }

    public function getUuidRegistry(): UuidRegistry
    {
        if (!isset($this->uuidRegistry)) {
            $this->uuidRegistry = new UuidRegistry(['table_name' => self::TABLE_NAME]);
        }
        return $this->uuidRegistry;
    }

    public function setUuidRegistry(UuidRegistry $uuidRegistry): void
    {
        $this->uuidRegistry = $uuidRegistry;
    }

    const DEFAULT_OB_STATUS = 'preliminary';

    public function getUuidFields(): array
    {
        return ['uuid', 'questionnaire_response_uuid', 'parent_observation_uuid', 'euuid', 'puuid', 'performer_uuid'];
    }

    public function getObservationTypeDisplayName($obType): string
    {
        if (!isset($this->typesById)) {
            $this->typesById = $this->getOptionsByListName('Observation_Types');
        }
        return $this->typesById[$obType] ?? $obType;
    }
    public function getStatusDisplayName($status): string
    {
        if (!isset($this->statiiById)) {
            $this->statiiById = $this->getOptionsByListName('observation-status');
        }
        return $this->statiiById[$status] ?? $status;
    }

    private function getOptionsByListName($listName): array
    {
        $options = $this->getListService()->getOptionsByListName($listName);
        $optionsById = [];
        foreach ($options as $option) {
            $optionsById[$option['option_id']] = $option['title'];
        }
        return $optionsById;
    }

    /**
     * Get observation data by form id including sub-observations
     * AI Generated: Enhanced to support parent-child observation relationships
     *
     * @param int $formId
     * @param int $pid
     * @param int $encounter
     * @return array
     */
    public function getObservationsByFormId(int $formId, int $pid, int $encounter): array
    {
        $sql = "SELECT * FROM `form_observation` WHERE id=? AND pid = ? AND encounter = ?";
        return QueryUtils::fetchRecords($sql, [$formId, $pid, $encounter]);
        $result = $this->search([
            'form_id' => $formId,
            'pid' => $pid,
            'encounter' => $encounter
        ]);
        if ($result->hasData()) {
            return $result->getData();
        }
        return [];
    }

    /**
     * Get observation by ID with enhanced questionnaire_response_id support
     * AI Generated: Enhanced to include questionnaire_response_id in results
     *
     * @param int $id
     * @param int $pid
     * @param bool $includeChildObservations Whether to include sub-observations
     * @return array|null
     */
    public function getObservationById(int $id, int $pid, bool $includeChildObservations = false): ?array
    {
        $search = ['id' => $id, 'pid' => $pid];
        $result = $includeChildObservations ? $this->searchAndPopulateChildObservations($search) : $this->search($search);
        if ($result->hasData()) {
            $observation = $result->getFirstDataResult();
        } else {
            return null;
        }

        // AI Generated: Add questionnaire response information if linked
        if (!empty($observation['questionnaire_response_id'])) {
            $observation['questionnaire_response'] = $this->getLinkedQuestionnaireResponse(
                $observation['questionnaire_response_id']
            );
        }

        return $observation;
    }

    /**
     * AI Generated: Get linked QuestionnaireResponse details
     *
     * @param int $questionnaireResponseId
     * @return array|null
     */
    private function getLinkedQuestionnaireResponse(int $questionnaireResponseId): ?array
    {
        $sql = "SELECT id, response_id, questionnaire_name, status, create_time, questionnaire_response
                FROM questionnaire_response
                WHERE id = ?";

        $records = QueryUtils::fetchRecords($sql, [$questionnaireResponseId]);

        if (empty($records)) {
            return null;
        }

        return $records[0];
    }

    /**
     * Get all observations for a patient/encounter with hierarchical structure
     * AI Generated: New method to support list view functionality
     *
     * @param int $pid
     * @param int $encounter
     * @return array
     */
    public function getAllObservationsForEncounter(int $pid, int $encounter): array
    {
        $sql = "SELECT * FROM `form_observation`
                WHERE pid = ? AND encounter = ? AND parent_observation_id IS NULL
                ORDER BY date DESC, id DESC";

        $mainObservations = QueryUtils::fetchRecords($sql, [$pid, $encounter]);

        // For each main observation, get its sub-observations
        foreach ($mainObservations as &$observation) {
            $observation['sub_observations'] = $this->getSubObservations($observation['id']);
        }

        return $mainObservations;
    }

    /**
     * @param $row
     * @return array
     */
    protected function createResultRecordFromDatabaseResult($row): array
    {
        $row['parent_observation_uuid'] ??= null;
        $record = (array)parent::createResultRecordFromDatabaseResult($row); // setup any uuid fields
        if (empty($record['ob_status'])) {
            $record['ob_status'] = self::DEFAULT_OB_STATUS; // default value
            $record['ob_status_display'] = $this->getStatusDisplayName($record['ob_status']);
        }
        if (!empty($record['sub_observations']) && is_array($record['sub_observations'])) {
            $record['sub_observations'] = array_map($this->createResultRecordFromDatabaseResult(...), $record['sub_observations']);
        }
        $record['ob_reason_status_display'] = $this->getReasonStatusDisplay($record['ob_reason_status']);
        return $record;
    }

    private function getReasonStatusDisplay($status): string
    {
        if (!isset($this->reasonCodes)) {
            $this->reasonCodes = ReasonStatusCodes::getCodesWithDescriptions();
        }
        return $this->reasonCodes[$status]['description'] ?? $status;
    }

    /**
     * Get sub-observations for a parent observation
     * AI Generated: New method to support Design 1 sub-observation functionality
     *
     * @param int $parentObservationId
     * @return array
     */
    public function getSubObservations(int $parentObservationId): array
    {
        $result = $this->search([
            'parent_observation_id' => $parentObservationId
        ]);
        if ($result->hasData()) {
            return $result->getData();
        } else {
            return [];
        }
    }

    public function searchAndPopulateChildObservations($search, $isAndCondition = true): ProcessingResult
    {
        $foundResult = $this->search($search);
        if (!$foundResult->hasData()) {
            return $foundResult;
        }

        // we need to loop through any found results and grab all records that do NOT have a parent_observation_uuid
        // because those are the top level observations, we will then search for the observation and their children
        // and replace the foundResult with the merged set
        $topLevelIds = [];
        $records = $foundResult->getData();
        $recordIndex = [];
        foreach ($records as $index => $result) {
            $recordIndex[$result['id']] = $index;
            if (empty($result['parent_observation_id'])) {
                $topLevelIds[] = (string)$result['id'];
            }
        }
        // this search has no top level observations so just return what we found
        if (empty($topLevelIds)) {
            return $foundResult;
        }
        // need to populate sub_observations for each top level observation
        $parentSearch = new TokenSearchField('parent_observation_id', $topLevelIds);
        $childRecordResult = $this->search([$parentSearch]);
        foreach ($childRecordResult->getData() as $childRecord) {
            if (isset($recordIndex[$childRecord['parent_observation_id']])) {
                $parentIndex = $recordIndex[$childRecord['parent_observation_id']];
                if (!isset($records[$parentIndex]['sub_observations'])) {
                    $records[$parentIndex]['sub_observations'] = [];
                }
                $records[$parentIndex]['sub_observations'][] = $childRecord;
            }
        }
        $finalResult = new ProcessingResult();
        $finalResult->setData($records);
        return $finalResult;
    }

    /**
     * @param ISearchField[]|string[]|array $search
     * @param bool $isAndCondition
     * @return ProcessingResult
     */
    public function search($search, $isAndCondition = true): ProcessingResult
    {
        if (empty($search['activity'])) {
            $search['activity'] = '1'; // default to active records only
        }
        // we first need to grab all of the record ids that match the search criteria
        // then we fetch all records where the id matches either the id or the parent_observation id in the list
        // so that we can grab the entire relationship.

        $sql = "SELECT
                fo.id, fo.uuid, fo.form_id, fo.date, fo.pid, fo.encounter, fo.user, fo.groupname, fo.authorized, fo.activity, fo.code
                ,fo.observation, fo.ob_value, fo.ob_unit, fo.description, fo.code_type, fo.table_code, fo.ob_code, fo.ob_type, fo.ob_status
                ,fo.result_status, fo.ob_reason_status, fo.ob_reason_code, fo.ob_documentationof_table, fo.ob_documentationof_table_id
                ,fo.date_end, fo.parent_observation_id, fo.category, fo.questionnaire_response_id
                ,fo.ob_value_code_description
                ,performer.performer_uuid
                ,performer.performer_type
                ,patient.puuid
                ,encounter.euuid
                ,observation_parent.parent_observation_uuid
                ,r.questionnaire_response_uuid
                ,r.questionnaire_name
                ,r.questionnaire_status
                ,r.questionnaire_date
                ,r.screening_category_code
                ,r.screening_category_display
                ,lo_type.ob_type_display
                ,lo_status.ob_status_display
                FROM form_observation fo
                LEFT JOIN (
                    SELECT
                        users.uuid AS performer_uuid
                        ,users.username AS performer_username
                        ,'Practitioner' AS performer_type
                    FROM users
                ) performer ON performer.performer_username = fo.user
                LEFT JOIN (
                    SELECT
                        title AS ob_type_display
                        ,option_id
                        ,list_id
                    FROM list_options
                ) lo_type ON (fo.ob_type = lo_type.option_id AND lo_type.list_id = 'Observation_Types')
                LEFT JOIN (
                    SELECT
                        title AS ob_status_display
                        ,option_id
                        ,list_id
                    FROM list_options
                ) lo_status ON (fo.ob_status = lo_status.option_id AND lo_status.list_id = 'observation-status')
                LEFT JOIN (
                    select
                        id AS parent_observation_fk_id
                        ,uuid AS parent_observation_uuid
                    FROM form_observation
                ) observation_parent ON observation_parent.parent_observation_fk_id = fo.parent_observation_id
                LEFT JOIN (
                    select
                        pid AS patient_pid
                        ,uuid AS puuid
                    FROM patient_data
                ) patient ON patient.patient_pid = fo.pid
                LEFT JOIN (
                    select
                        encounter AS eid
                        ,uuid AS euuid
                    FROM form_encounter
                ) encounter ON encounter.eid = fo.encounter
                LEFT JOIN (
                    SELECT
                        resp.id AS response_id
                        ,resp.questionnaire_name
                        ,resp.status AS questionnaire_status
                        ,resp.create_time AS questionnaire_date
                        ,resp.uuid AS questionnaire_response_uuid
                        ,lo.option_id AS screening_category_code
                        ,lo.title AS screening_category_display
                    FROM questionnaire_response resp
                    LEFT JOIN questionnaire_repository quest ON (quest.id = resp.questionnaire_foreign_id)
                    LEFT JOIN list_options lo ON (lo.option_id = quest.category AND lo.list_id = 'Observation_Types')
                ) r ON fo.questionnaire_response_id = r.response_id ";
        $whereFragment = FhirSearchWhereClauseBuilder::build($search, $isAndCondition);
        $sql .= $whereFragment->getFragment();
        // TODO: @adunsulag need to implement pagination and sorting
        $sql .= " ORDER BY fo.date DESC, fo.parent_observation_id, fo.id";
        $records = QueryUtils::fetchRecords($sql, $whereFragment->getBoundValues());
        $result = new ProcessingResult();
        // make one final pass to create result records
        // child records will now be part of the parent records and set to null so we'll make O(n) passes
        foreach ($records as $record) {
            if (empty($record)) {
                continue;
            }
            $result->addData($this->createResultRecordFromDatabaseResult($record));
        }
        return $result;
    }

    /**
     * Get the next available form ID
     *
     * @return int
     */
    public function getNextFormId(): int
    {
        $getMaxid = QueryUtils::fetchSingleValue("SELECT MAX(id) as largestId FROM `form_observation`", 'largestId');
        if ($getMaxid != null) {
            return intval($getMaxid) + 1;
        }

        return 1;
    }

    /**
     * @param int $id Observation id to delete
     * @param int $formId Form id to ensure we don't delete other observations in this form.
     * @param int $pid Patient id to ensure we don't delete other patients data
     * @param int $encounter Encounter id to ensure we don't delete other encounters data
     * @return void
     */
    public function deleteObservationById(int $id, int $formId, int $pid, int $encounter): void
    {
        // we don't delete the records, we just set the activity to be 0
        QueryUtils::sqlStatementThrowException(
            "UPDATE `form_observation` SET `activity`=0 WHERE (id =? OR parent_observation_id = ?) AND pid = ? AND encounter = ?",
            [$id, $id, $pid, $encounter]
        );
    }

    /**
     * Save observation data with enhanced schema support
     * AI Generated: Updated to use new database schema fields
     *
     * @param array $observationData
     * @return array The updated observation data including the ID
     */
    private function saveObservationRecord(array $observationData): array
    {
        $sets = "`form_id` = ?,
            `uuid`        = ?,
            `pid`         = ?,
            `groupname`   = ?,
            `user`        = ?,
            `encounter`   = ?,
            `authorized`  = ?,
            `activity`    = 1,
            `observation` = ?,
            `code`        = ?,
            `code_type`   = ?,
            `description` = ?,
            `table_code`  = ?,
            `ob_type`     = ?,
            `ob_code`     = ?,
            `ob_status`   = ?,
            `ob_value`    = ?,
            `ob_unit`     = ?,
            `date`        = ?,
            `ob_reason_code` = ?,
            `ob_reason_status` = ?,
            `ob_reason_text` = ?,
            `date_end` = ?,
            `parent_observation_id` = ?,
            `category` = ?,
            `questionnaire_response_id` = ?";

        if (empty($observationData['uuid'])) {
            $observationData['uuid'] = $this->getUuidRegistry()->createUuid();
        } else {
            $observationData['uuid'] = UuidRegistry::uuidToBytes($observationData['uuid']);
        }
        $encounter = $observationData['encounter'];
        $pid = $observationData['pid'];
        $userauthorized = $observationData['authorized'] ?? 0;
        if (empty($observationData['form_id'])) {
            $observationData['form_id'] = $this->getNextFormId();
            // we need to add the form to the encounter
            // TODO: @adunsulag we need to dispatch save events here for module writers.
            $this->getFormService()->addForm(
                $encounter,
                self::FORM_NAME,
                $observationData['form_id'],
                self::FORM_DIR,
                $pid,
                $userauthorized
            );
        }
        // code needs to match code_type
        $codeService = $this->getCodeTypesService();
        $code = $codeService->parseCode($observationData['code']);
        $observationData['ob_code'] = $code['code'] ?? '';
        // observations default to LOINC if not specified
        $observationData['code_type'] = $code['code_type'] ?? 'LOINC';
        $sqlBindArray = [
            $observationData['form_id'],
            $observationData['uuid'],
            $pid,
            $observationData['groupname'],
            $observationData['user'],
            $encounter,
            $userauthorized,
            $observationData['observation'] ?? '',
            $observationData['code'] ?? '',
            $observationData['code_type'] ?? '',
            $observationData['description'] ?? '',
            $observationData['table_code'] ?? '',
            $observationData['ob_type'] ?? '',
            $observationData['ob_code'] ?? '',
            $observationData['ob_status'] ?? '',
            $observationData['ob_value'] ?? '',
            $observationData['ob_unit'] ?? '',
            $observationData['date'] ?? date('Y-m-d H:i:s'),
            $observationData['ob_reason_code'] ?? '',
            $observationData['ob_reason_status'] ?? '',
            $observationData['ob_reason_text'] ?? '',
            $observationData['date_end'] ?? null,
            $observationData['parent_observation_id'] ?? null,
            $observationData['category'] ?? null,
            $observationData['questionnaire_response_id'] ?? null
        ];

        if (!empty($observationData['id'])) {
            $sql = "UPDATE `form_observation` SET $sets WHERE id = ? AND pid = ? AND encounter = ?";
            $sqlBindArray[] = $observationData['id'];
            // we make sure updates are not done to other patients/encounters
            $sqlBindArray[] = $pid;
            $sqlBindArray[] = $encounter;
            QueryUtils::sqlStatementThrowException(
                $sql,
                $sqlBindArray
            );
        } else {
            $sql = "INSERT INTO `form_observation` SET $sets";
            $observationData['id'] = QueryUtils::sqlInsert(
                $sql,
                $sqlBindArray
            );
        }
        // fetch the saved data and return it, that way dates, and everything else is in correct format
        $dbObservation = $this->getObservationById($observationData['id'], $pid);
        if (empty($dbObservation)) {
            throw new RuntimeException("Failed to fetch saved observation record for id: " . $observationData['id']);
        }
        return $dbObservation;
    }

    /**
     * Get observation types from list options
     *
     * @return array
     */
    public function getObservationTypes(): array
    {
        return QueryUtils::fetchRecords("SELECT `option_id`, `title` FROM `list_options` WHERE `list_id` = 'Observation_Types' ORDER BY `seq`");
    }

    /**
     * Save main observation with sub-observations
     *
     * @param array $mainObservationData
     * @param array $subObservationsData
     * @return array The updated observation data including sub-observations
     */
    public function saveObservation(array $observation): array
    {
        // TODO: @adunsulag should we validate here or let controller handle it?

        // Save main observation first
        $savedObservation = $this->saveObservationRecord($observation);
        $subObservationsData = $observation['sub_observations'] ?? [];
        // Save sub-observations
        $savedSubs = [];
        foreach ($subObservationsData as $subObsData) {
            $subObsData['parent_observation_id'] = $savedObservation['id'];
            $subObsData['pid'] = $savedObservation['pid'];
            $subObsData['encounter'] = $savedObservation['encounter'];
            $subObsData['form_id'] = $savedObservation['form_id']; // make sure sub-observations have same form_id

            $savedSubs[] = $this->saveObservationRecord($subObsData);
        }
        $savedObservation['sub_observations'] = $savedSubs;
        return $savedObservation;
    }

    /**
     * Validate observation data before saving
     * AI Generated: New method for data validation
     *
     * @param array $observationData
     * @return array Array of validation errors (empty if valid)
     */
    public function validateObservationData(array $observationData): array
    {
        $errors = [];

        if (empty($observationData['pid'])) {
            $errors[] = 'Patient ID is required';
        }
        if (empty($observationData['encounter'])) {
            $errors[] = 'Encounter ID is required';
        }
        if (empty($observationData['groupname'])) {
            $errors[] = 'Group name is required';
        }

        if (empty($observationData['user'])) {
            $errors[] = 'User is required';
        }
        // form_id can be 0 or positive integer
        if (!isset($observationData['form_id'])) {
            $errors[] = 'Form ID is required';
        }

        // Required field validation
        if (empty($observationData['code'])) {
            $errors[] = 'Code is required';
        }

        if (empty($observationData['description'])) {
            $errors[] = 'Description is required';
        }

        if (empty($observationData['date'])) {
            $errors[] = 'Date is required';
        }

        // Date format validation
        $dateTime = null;
        if (!empty($observationData['date'])) {
            $dateTime = DateFormatterUtils::dateStringToDateTime($observationData['date']);
            if ($dateTime === false) {
                $errors[] = 'Invalid date format';
            }
        }

        // End date validation
        if (!empty($observationData['date_end'])) {
            $endDateTime = DateFormatterUtils::dateStringToDateTime($observationData['date']);
            if ($endDateTime === false) {
                $errors[] = 'Invalid end date format';
            } elseif (!empty($dateTime)) {
                if ($endDateTime < $dateTime) {
                    $errors[] = 'End date cannot be before start date';
                }
            }
        }

        // Category validation
        if (!empty($observationData['ob_type'])) {
            $optionId = $this->getListService()->getListOption('Observation_Types', $observationData['ob_type']);
            if (empty($optionId)) {
                $errors[] = 'Invalid type specified';
            }
        }

        // AI Generated: Validate questionnaire_response_id if provided
        if (!empty($observationData['questionnaire_response_id'])) {
            if (!is_numeric($observationData['questionnaire_response_id'])) {
                $errors[] = xl('Invalid questionnaire response ID format');
            } else {
                // Check if the questionnaire response exists
                if (!$this->questionnaireResponseExists($observationData['questionnaire_response_id'])) {
                    $errors[] = xl('Selected questionnaire response does not exist');
                }
            }
        }
        // end AI Generated

        return $errors;
    }


    /**
     * AI Generated: Check if questionnaire response exists in database
     *
     * @param int $questionnaireResponseId
     * @return bool
     */
    private function questionnaireResponseExists(int $questionnaireResponseId): bool
    {
        $sql = "SELECT COUNT(*) as count FROM questionnaire_response WHERE id = ?";
        $result = QueryUtils::fetchSingleValue($sql, 'count', [$questionnaireResponseId]);
        return $result > 0;
    }

    /**
     * Search observations by criteria
     * AI Generated: New method to support filtering in list view
     *
     * @param int $pid
     * @param int $encounter
     * @param array $searchCriteria
     * @return array
     */
    public function searchObservations(int $pid, int $encounter, array $searchCriteria = []): array
    {
        $sql = "SELECT * FROM `form_observation`
                WHERE pid = ? AND encounter = ? AND parent_observation_id IS NULL";

        $params = [$pid, $encounter];

        if (!empty($searchCriteria['form_id'])) {
            $sql .= " AND form_id = ? ";
            $params[] = $searchCriteria['form_id'];
        }

        // Add search filters
        if (!empty($searchCriteria['search_term'])) {
            $sql .= " AND (code LIKE ? OR description LIKE ? OR observation LIKE ?)";
            $searchTerm = '%' . $searchCriteria['search_term'] . '%';
            $params[] = $searchTerm;
            $params[] = $searchTerm;
            $params[] = $searchTerm;
        }

        if (!empty($searchCriteria['ob_type'])) {
            $sql .= " AND ob_type = ?";
            $params[] = $searchCriteria['ob_type'];
        }

        if (!empty($searchCriteria['ob_type'])) {
            $sql .= " AND ob_type = ?";
            $params[] = $searchCriteria['ob_type'];
        }

        if (!empty($searchCriteria['date_range'])) {
            switch ($searchCriteria['date_range']) {
                case 'today':
                    $sql .= " AND DATE(date) = CURDATE()";
                    break;
                case 'week':
                    $sql .= " AND date >= DATE_SUB(NOW(), INTERVAL 7 DAY)";
                    break;
                case 'month':
                    $sql .= " AND date >= DATE_SUB(NOW(), INTERVAL 30 DAY)";
                    break;
                // 'all' or default - no additional filter
            }
        }

        $sql .= " ORDER BY date DESC, id DESC";

        $mainObservations = QueryUtils::fetchRecords($sql, $params);
        $mainObservations = array_map($this->createResultRecordFromDatabaseResult(...), $mainObservations ?? []);

        // For each main observation, get its sub-observations
        foreach ($mainObservations as &$observation) {
            $observation['sub_observations'] = $this->getSubObservations($observation['id']);
        }

        return $mainObservations;
    }

    public function getNewObservationTemplate(): array
    {
        return [
            'id' => '',
            'code' => '',
            'description' => '',
            'ob_value' => '',
            'ob_unit' => '',
            'ob_status' => self::DEFAULT_OB_STATUS,
            'date' => date('Y-m-d H:i:s'),
            'date_end' => '',
            'code_type' => '',
            'table_code' => '',
            'ob_type' => '',
            'observation' => '',
            'questionnaire_response_id' => null,
            'ob_reason_code' => '',
            'ob_reason_status' => '',
            'ob_reason_text' => '',
            'parent_observation_id' => null
        ];
    }
}
