<?php

/**
 * ResourceServiceSearchTrait handles the creating of openemr search parameters for a resource.
 *
 * @package openemr
 * @link      http://www.open-emr.org
 * @author    Stephen Nielson <snielson@discoverandchange.com>
 * @copyright Copyright (c) 2022 Discover and Change <snielson@discoverandchange.com>
 * @license   https://github.com/openemr/openemr/blob/master/LICENSE GNU General Public License 3
 */

namespace OpenEMR\Services\FHIR\Traits;

use OpenEMR\Services\FHIR\IPatientCompartmentResourceService;
use OpenEMR\Services\Search\FHIRSearchFieldFactory;
use OpenEMR\Services\Search\ISearchField;
use OpenEMR\Services\Search\SearchFieldException;
use OpenEMR\Services\Search\SearchFieldOrder;
use InvalidArgumentException;

trait ResourceServiceSearchTrait
{
    /**
     * @var FHIRSearchFieldFactory
     */
    private FHIRSearchFieldFactory $searchFieldFactory;

    public function setSearchFieldFactory(FHIRSearchFieldFactory $factory): void
    {
        $this->searchFieldFactory = $factory;
    }

    public function getSearchFieldFactory(): FHIRSearchFieldFactory
    {
        return $this->searchFieldFactory;
    }

    /**
     * Given the hashmap of search parameters to values it generates a map of search keys to ISearchField objects that
     * are used to search in the OpenEMR system.  Service classes that extend the base class can override this method
     *
     * to either add search fields or change the functionality of the created ISearchFields.
     *
     * @param array $fhirSearchParameters
     * @param string|null $puuidBind The patient unique id if searching in a patient context
     * @return ISearchField[] where the keys are the search fields.
     */
    protected function createOpenEMRSearchParameters(array $fhirSearchParameters, ?string $puuidBind = null): array
    {
        $oeSearchParameters = [];

        $specialColumns = ['_sort' => '', '_count' => '', '_offset' => ''];
        $hasSort = false;

        foreach ($fhirSearchParameters as $fhirSearchField => $searchValue) {
            try {
                if (isset($specialColumns[$fhirSearchField])) {
                    $config = $oeSearchParameters['_config'] ?? [];
                    if ($fhirSearchField == '_sort') {
                        $hasSort = true;
                    }
                    $config[$fhirSearchField] = $searchValue;
                    $oeSearchParameters['_config'] = $config;
                    continue;
                }
                // format: <field>{:modifier1|:modifier2}={comparator1|comparator2}[value1{,value2}]
                // field is the FHIR search field
                // modifier is the search modifier ie :exact, :contains, etc
                // comparator is used with dates / numbers, ie :gt, :lt
                // values can be comma separated and are treated as an OR condition
                // if the $searchValue is an array then this is treated as an AND condition
                // if $searchValue is an array and individual fields contains comma separated values the and clause takes
                // precedence and ALL values will be UNIONED (AND clause).
                $searchField = $this->createSearchParameterForField($fhirSearchField, $searchValue);
                $oeSearchParameters[$searchField->getName()] = $searchField;
            } catch (InvalidArgumentException $exception) {
                $message = "The search field argument was invalid, improperly formatted, or could not be parsed. "
                    . " Inner message: " . $exception->getMessage();
                throw new SearchFieldException($fhirSearchField, $message, $exception->getCode(), $exception);
            }
        }

        // now that we've created all of our fields, let's go through and create our sort
        if ($hasSort) {
            $oeSearchParameters['_config']['_sort'] = $this->createSortParameter($fhirSearchParameters['_sort']);
        }

        // we make sure if we are a resource that deals with patient data and we are in a patient bound context that
        // we restrict the data to JUST that patient.
        if (!empty($puuidBind) && $this instanceof IPatientCompartmentResourceService) {
            $searchFactory = $this->getSearchFieldFactory();
            $patientField = $this->getPatientContextSearchField();
            // TODO: @adunsulag not sure if every service will already have a defined binding for the patient... I'm assuming for Patient compartments we would...
            // yet we may need to extend the factory in the future to handle this.
            $oeSearchParameters[$patientField->getName()] = $searchFactory->buildSearchField($patientField->getName(), [$puuidBind]);
        }

        return $oeSearchParameters;
    }

    private function createSortParameter($sort): array
    {
        $newSortOrder = [];
        $sortFields = explode(',', (string) $sort);
        $searchFactory = $this->getSearchFieldFactory();
        foreach ($sortFields as $sortField) {
            $isDescending = ($sortField[0] ?? '') === '-';
            if ($isDescending) {
                $sortField = substr($sortField, 1);
            }

            // TODO: @adunsulag would it be more efficient to just build the
            if ($searchFactory->hasSearchField($sortField)) {
                $definition = $searchFactory->getSearchFieldDefinition($sortField);
                $mappedFields = $definition->getMappedFields();
                foreach ($mappedFields as $field) {
                    $newSortOrder[] = new SearchFieldOrder($field, !$isDescending);
                }
            }
        }
        return $newSortOrder;
    }

    protected function createSearchParameterForField($fhirSearchField, $searchValue): ISearchField
    {
        $searchFactory = $this->getSearchFieldFactory();
        if ($searchFactory->hasSearchField($fhirSearchField)) {
            return $searchFactory->buildSearchField($fhirSearchField, $searchValue);
        } else {
            throw new SearchFieldException($fhirSearchField, xlt("This search field does not exist or is not supported"));
        }
    }
}
