<?php

/*
 * OpenEMRFhirQuestionnaireResponse.php
 * @package openemr
 * @link      http://www.open-emr.org
 * @author    Stephen Nielson <snielson@discoverandchange.com>
 * @copyright Copyright (c) 2025 Stephen Nielson <snielson@discoverandchange.com>
 * @license   https://github.com/openemr/openemr/blob/master/LICENSE GNU General Public License 3
 */

namespace OpenEMR\FHIR\DomainModels;

use OpenEMR\FHIR\R4\FHIRDomainResource\FHIRQuestionnaireResponse;
use OpenEMR\FHIR\R4\FHIRElement\FHIRCanonical;
use OpenEMR\FHIR\R4\FHIRElement\FHIRExtension;
use OpenEMR\FHIR\R4\FHIRElement\FHIRReference;
use OpenEMR\FHIR\R4\FHIRResource\FHIRQuestionnaireResponse\FHIRQuestionnaireResponseAnswer;
use OpenEMR\FHIR\R4\FHIRResource\FHIRQuestionnaireResponse\FHIRQuestionnaireResponseItem;
use SimpleXMLElement;

class OpenEMRFhirQuestionnaireResponse extends FHIRQuestionnaireResponse
{
    /**
     * Note that _questionnaire is a primitive type override that allows for questionnaire to be overridden, newer versions of FHIR R4
     * php-carbon support this but until we can update all of the FHIR DomainElements we are going to support Us Core 8.0
     * this way.  Note FHIRCanonical has an extension property that allows it to be extended.
     * @var FHIRCanonical|null
     */

    public function __construct($data = [])
    {
        if (!empty($data['author']['reference'])) {
            $data['author'] = new FHIRReference($data['author']);
        }
        if (!empty($data['item'])) {
            $data['item'] = $this->parseQuestionnaireResponseItems($data);
        }
        if (!empty($data['subject'])) {
            $data['subject'] = new FHIRReference($data['subject']);
        }
        // now handle the rest since we've properly formatted it now.
        parent::__construct($data);

        // setup our overridden _questionnaire attribute if we have it.
        if (isset($data['_questionnaire'])) {
            // will have the extension attribute and everything needed
            if (isset($data['_questionnaire']['extension'])) {
                $extensions = [];
                foreach ($data['_questionnaire']['extension'] as $extension) {
                    $extensions[] = new FHIRExtension($extension);
                }
                $data['_questionnaire']['extension'] = $extensions;
            }
            $value = new FHIRCanonical($data['_questionnaire']);
            if (isset($this->questionnaire)) {
                $value->setValue(is_string($this->questionnaire) ? $this->questionnaire : $this->questionnaire->getValue());
            }
            $this->questionnaire = $value;
        }
    }

    public function jsonSerialize(): mixed
    {
        $json = parent::jsonSerialize();
        if (isset($this->questionnaire)) {
            if (0 < count($this->questionnaire->getExtension())) {
                $json['_questionnaire'] = [
                    'extension' => $this->questionnaire->getExtension()
                ];
            }
        }
        return $json;
    }

    public function xmlSerialize($returnSXE = false, $sxe = null)
    {
        parent::xmlSerialize($returnSXE, $sxe); // TODO: Change the autogenerated stub
        if (null === $sxe) {
            $sxe = new SimpleXMLElement('<QuestionnaireResponse xmlns="http://hl7.org/fhir"></QuestionnaireResponse>');
        }
        parent::xmlSerialize(true, $sxe);
        if (isset($this->questionnaire) && 0 < count($this->questionnaire->getExtension())) {
            foreach ($this->questionnaire->getExtension() as $questionnaireExtension) {
                $questionnaireExtension->xmlSerialize(true, $sxe->addChild('_questionnaire'));
            }
        }

        if ($returnSXE) {
            return $sxe;
        }
        return $sxe->saveXML();
    }

    protected function parseQuestionnaireResponseItems(array $dataItem)
    {
        $objItems = [];
        if (!empty($dataItem['item'])) {
            foreach ($dataItem['item'] as $item) {
                if (!empty($item['item'])) {
                    $item['item'] = $this->parseQuestionnaireResponseItems($item);
                }
                if (!empty($item['answer'])) {
                    $answers = [];
                    foreach ($item['answer'] as $answer) {
                        if (!empty($answer['item'])) {
                            $answer['item'] = $this->parseQuestionnaireResponseItems($answer['item']);
                        }
                        $answers[] = new FHIRQuestionnaireResponseAnswer($answer);
                    }
                    $item['answer'] = $answers;
                }
                $item = new FHIRQuestionnaireResponseItem($item);
                $objItems[] = $item;
            }
        }
        return $objItems;
    }
}
