<?php

/*
 * ResourceScopeEntityList.php
 * @package openemr
 * @link      http://www.open-emr.org
 * @author    Stephen Nielson <snielson@discoverandchange.com>
 * @copyright Copyright (c) 2025 Stephen Nielson <snielson@discoverandchange.com>
 * @license   https://github.com/openemr/openemr/blob/master/LICENSE GNU General Public License 3
 */

namespace OpenEMR\Common\Auth\OpenIDConnect\Entities;

use League\OAuth2\Server\Entities\Traits\EntityTrait;
use ArrayObject;

class ResourceScopeEntityList extends ArrayObject
{
    use EntityTrait;

    public function __construct(string $identifier, object|array $array = [], int $flags = 0, string $iteratorClass = "ArrayIterator")
    {
        $this->setIdentifier($identifier);
        foreach ($array as $item) {
            if (!$item instanceof ScopeEntity) {
                throw new \InvalidArgumentException('All items must be instances of ScopeEntityInterface');
            }
            if ($identifier !== $item->getScopeLookupKey()) {
                throw new \InvalidArgumentException('All items must have the same identifier as the list');
            }
        }
        parent::__construct($array, $flags, $iteratorClass);
    }

    public function append(mixed $item): void
    {
        $this->validateItem($item);
        parent::append($item); // TODO: Change the autogenerated stub
    }

    public function offsetSet(mixed $key, mixed $value): void
    {
        $this->validateItem($value);
        parent::offsetSet($key, $value); // TODO: Change the autogenerated stub
    }

    private function validateItem(mixed $item): void
    {
        if (!$item instanceof ScopeEntity) {
            throw new \InvalidArgumentException('Item must be an instance of ScopeEntityInterface');
        }
        if ($this->getIdentifier() !== $item->getScopeLookupKey()) {
            throw new \InvalidArgumentException('Item must have the same identifier as the list');
        }
    }

    public function containsScope(ScopeEntity $scope): bool
    {
        foreach ($this as $item) {
            /**
             * @var ScopeEntity $item
             */
            if ($item->containsScope($scope)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Returns an array of scopes that contain the given scope.  This is useful for finding scopes that are more specific than the given scope such
     * as scopes that have a specific granular restriction.
     * For example contained scopes for user/Condition.r would include user/Condition.rs &&
     * user/Condition.rs?category=http://hl7.org/fhir/us/core/CodeSystem/condition-category|health-concern
     * @param ScopeEntity $scope
     * @return array
     */
    public function getContainedScopes(ScopeEntity $scope): array
    {
        $containedScopes = [];
        foreach ($this as $item) {
            /**
             * @var ScopeEntity $item
             */
            if ($item->containsScope($scope)) {
                $containedScopes[] = $item;
            }
        }
        return $containedScopes;
    }
}
