<?php

/**
 * Module is responsible for setting up the configuration of the module and any events it listens to.
 *
 * @package   OpenEMR
 * @link      https://www.open-emr.org
 * @author    Vinish K <vinish@zhservices.com>
 * @author    Stephen Nielson <snielson@discoverandchange.com>
 * @author    Michael A. Smith <michael@opencoreemr.com>
 * @copyright Copyright (c) 2015 Z&H Consultancy Services Private Limited <sam@zhservices.com>
 * @copyright Copyright (c) 2022 Discover and Change <snielson@discoverandchange.com>
 * @copyright Copyright (c) 2026 OpenCoreEMR Inc <https://opencoreemr.com/>
 * @license   https://github.com/openemr/openemr/blob/master/LICENSE GNU General Public License 3
 */

namespace Carecoordination;

use Laminas\ModuleManager\ModuleManager;
use Laminas\Mvc\MvcEvent;
use Laminas\View\Helper\Openemr\Emr;
use Laminas\View\Helper\Openemr\Menu;
use Carecoordination\Model\Progressnote;
use Carecoordination\Model\ProgressnoteTable;
use Carecoordination\Model\Continuitycaredocument;
use Carecoordination\Model\ContinuitycaredocumentTable;
use Carecoordination\Listener\CCDAEventsSubscriber;
use OpenEMR\Common\Acl\AclMain;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

class Module
{
    public function getAutoloaderConfig()
    {
        return [
            \Laminas\Loader\ClassMapAutoloader::class => [
                __DIR__ . '/autoload_classmap.php',
            ],
            \Laminas\Loader\StandardAutoloader::class => [
                'namespaces' => [
                    __NAMESPACE__ => __DIR__ . '/src/' . __NAMESPACE__,

                ],
            ],
        ];
    }

    public function getConfig()
    {
        return include __DIR__ . '/config/module.config.php';
    }

    public function init(ModuleManager $moduleManager)
    {
        $sharedEvents = $moduleManager->getEventManager()->getSharedManager();
        $sharedEvents->attach(__NAMESPACE__, 'dispatch', function ($e): void {
            // Enforce ACL for the Care Coordination module.
            // Skip for patient portal sessions which have their own authorization.
            if (($_SESSION['sessionUser'] ?? '') !== '-patient-') {
                $userId = $_SESSION['authUserID'] ?? '';
                if (
                    !AclMain::zhAclCheck($userId, 'send_to_hie')
                    && !AclMain::aclCheckCore('admin', 'super')
                ) {
                    echo xlt('Not Authorized');
                    exit;
                }
            }

            $controller = $e->getTarget();
            $controller->layout('carecoordination/layout/layout');
                $route = $controller->getEvent()->getRouteMatch();
                $controller->getEvent()->getViewModel()->setVariables([
                    'current_controller' => $route->getParam('controller'),
                    'current_action' => $route->getParam('action'),
                ]);
        }, 100);
    }

    public function onBootstrap(MvcEvent $e)
    {
        // we grab the OpenEMR event listener (which is injected as Laminas has its own dispatcher)
        $serviceManager = $e->getApplication()->getServiceManager();
        $oemrDispatcher = $serviceManager->get(EventDispatcherInterface::class);

        // now we can listen to our module events
        $menuSubscriber = $serviceManager->get(CCDAEventsSubscriber::class);
        $oemrDispatcher->addSubscriber($menuSubscriber);
    }
}
