<?php

/*
 * FhirQuestionnaireResponseFormServiceIntegrationTest
 *
 * @package openemr
 * @link      http://www.open-emr.org
 * @author    Stephen Nielson <snielson@discoverandchange.com>
 * @copyright Public Domain for sections that were generated with Claude.AI on September 3rd 2025
 * @license   https://github.com/openemr/openemr/blob/master/LICENSE GNU General Public License 3
 */

namespace OpenEMR\Tests\Services\FHIR\QuestionnaireResponse;

use Monolog\Level;
use OpenEMR\Common\Database\QueryUtils;
use OpenEMR\Common\Logging\SystemLogger;
use OpenEMR\Common\Uuid\UuidRegistry;
use OpenEMR\FHIR\R4\FHIRDomainResource\FHIRQuestionnaireResponse;
use OpenEMR\Services\FHIR\QuestionnaireResponse\FhirQuestionnaireResponseFormService;
use OpenEMR\Services\PatientService;
use OpenEMR\Services\QuestionnaireService;
use OpenEMR\Services\QuestionnaireResponseService;
use PHPUnit\Framework\TestCase;
use Exception;

/**
 * Integration tests for FhirQuestionnaireResponseFormService ensuring US Core 8.0 compliance with database operations
 */
class FhirQuestionnaireResponseFormServiceIntegrationTest extends TestCase
{
    // AI Generated by Claude.AI
    const QUESTIONNAIRE_NAME = 'PHPUnit Integration Test Questionnaire';
    const QUESTIONNAIRE_NAME_2 = 'PHPUnit Integration Test Questionnaire 2';

    /**
     * @var array Saved off session storage while we run this test.
     */
    private array $originalSession;

    /**
     * @var FhirQuestionnaireResponseFormService
     */
    private FhirQuestionnaireResponseFormService $service;

    /**
     * @var array Test patient data
     */
    private array $testPatientData;

    /**
     * @var string Test questionnaire UUID
     */
    private string $testQuestionnaireUuid;

    /**
     * @return void
     * @throws Exception
     */
    protected function setUp(): void
    {
        parent::setUp();

        // Store original session
        $this->originalSession = $_SESSION;
        $_SESSION['authUserID'] = QueryUtils::fetchSingleValue('select id FROM users ORDER BY id LIMIT 1', 'id');

        $this->service = new FhirQuestionnaireResponseFormService();
        $this->service->setSystemLogger(new SystemLogger(Level::Critical));

        // Create test patient - AI Generated test data creation
        $this->createTestPatient();

        // Create test questionnaire - AI Generated questionnaire setup
        $this->createTestQuestionnaire();
    }

    protected function tearDown(): void
    {
        parent::tearDown();

        // Restore session
        $_SESSION = $this->originalSession;

        // Clean up database records - AI Generated cleanup
        $this->cleanupTestData();
    }

    /**
     * Test that getOne() returns a US Core 8.0 compliant QuestionnaireResponse
     * @throws Exception
     */
    public function testGetOneReturnsUSCoreCompliantQuestionnaireResponse(): void
    {
        // Create a QuestionnaireResponse in the database - AI Generated test data
        $responseUuid = $this->createTestQuestionnaireResponse();

        // Retrieve using the service
        $processingResult = $this->service->getOne($responseUuid);

        $this->assertTrue($processingResult->isValid(), "Processing result should be valid: " . print_r($processingResult->getValidationMessages(), true));
        $this->assertNotEmpty($processingResult->getData(), "Processing result should contain data");
        $this->assertCount(1, $processingResult->getData(), "Should return exactly one QuestionnaireResponse");

        $questionnaireResponse = $processingResult->getFirstDataResult();
        $this->assertInstanceOf(FHIRQuestionnaireResponse::class, $questionnaireResponse);

        // Verify US Core 8.0 compliance
        $this->assertUSCoreCompliance($questionnaireResponse);

        // Verify the UUID matches
        $this->assertEquals($responseUuid, $questionnaireResponse->getId()->getValue());
    }

    /**
     * Test that getAll() returns US Core 8.0 compliant QuestionnaireResponses
     * @throws Exception
     */
    public function testGetAllReturnsUSCoreCompliantQuestionnaireResponses(): void
    {
        // Create multiple QuestionnaireResponses - AI Generated multiple test records
        $this->createTestQuestionnaireResponse();
        $this->createTestQuestionnaireResponse('in-progress');

        // Retrieve all using the service
        $processingResult = $this->service->getAll(['patient' => $this->testPatientData['uuid']]);

        $this->assertTrue($processingResult->isValid(), "Processing result should be valid");
        $this->assertNotEmpty($processingResult->getData(), "Processing result should contain data");
        $this->assertGreaterThanOrEqual(2, count($processingResult->getData()), "Should return at least the two test responses");

        // Verify each response is US Core compliant
        foreach ($processingResult->getData() as $questionnaireResponse) {
            $this->assertInstanceOf(FHIRQuestionnaireResponse::class, $questionnaireResponse);
            $this->assertUSCoreCompliance($questionnaireResponse);
        }
    }

    /**
     * Test search functionality with patient filtering
     * @throws Exception
     */
    public function testSearchByPatientReturnsUSCoreCompliantResults(): void
    {
        // Create QuestionnaireResponse for our test patient - AI Generated patient-specific test
         $this->createTestQuestionnaireResponse();

        // Search by patient UUID
        $searchParams = ['patient' => $this->testPatientData['uuid']];
        $processingResult = $this->service->getAll($searchParams);

        $this->assertTrue($processingResult->isValid(), "Search processing result should be valid");
        $this->assertNotEmpty($processingResult->getData(), "Search should return data");

        foreach ($processingResult->getData() as $questionnaireResponse) {
            $this->assertInstanceOf(FHIRQuestionnaireResponse::class, $questionnaireResponse);
            $this->assertUSCoreCompliance($questionnaireResponse);

            // Verify the subject references our test patient
            $subjectRef = $questionnaireResponse->getSubject()->getReference();
            $this->assertEquals("Patient/" . $this->testPatientData['uuid'], $subjectRef);
        }
    }

    /**
     * Test search functionality with questionnaire filtering
     * @throws Exception
     */
    public function testSearchByQuestionnaireReturnsUSCoreCompliantResults(): void
    {
        // Create QuestionnaireResponse - AI Generated questionnaire-specific test
        $this->createTestQuestionnaireResponse();

        // Search by questionnaire reference
        $searchParams = ['questionnaire' => $this->testQuestionnaireUuid];
        $processingResult = $this->service->getAll($searchParams);

        $this->assertTrue($processingResult->isValid(), "Questionnaire search should be valid");
        $this->assertNotEmpty($processingResult->getData(), "Questionnaire search should return data");

        foreach ($processingResult->getData() as $questionnaireResponse) {
            $this->assertInstanceOf(FHIRQuestionnaireResponse::class, $questionnaireResponse);
            $this->assertUSCoreCompliance($questionnaireResponse);

            // Verify the questionnaire reference matches
            $questionnaireRef = $questionnaireResponse->getQuestionnaire()->getValue();
            $this->assertStringContainsString($this->testQuestionnaireUuid, $questionnaireRef);
        }
    }

    /**
     * Test that parseOpenEMRRecord handles real database data correctly
     * @throws Exception
     */
    public function testParseOpenEMRRecordWithRealDatabaseData(): void
    {
        // Create a real QuestionnaireResponse in database - AI Generated real data test
        $responseUuid = $this->createTestQuestionnaireResponse();

        // Retrieve raw database record
        $questionnaireResponseService = new QuestionnaireResponseService();
        $dbResult = $questionnaireResponseService->search(['questionnaire_response_uuid' => UuidRegistry::uuidToBytes($responseUuid)]);

        $this->assertTrue($dbResult->isValid(), "Database retrieval should be valid");
        $this->assertNotEmpty($dbResult->getData(), "Database should return data");

        $rawData = $dbResult->getFirstDataResult();

        // Parse using the service
        $parsedResponse = $this->service->parseOpenEMRRecord($rawData);
        $this->assertUSCoreCompliance($parsedResponse);

        // Verify the parsed data matches the original
        $this->assertEquals($responseUuid, $parsedResponse->getId()->getValue());
    }

    /**
     * Test handling of different QuestionnaireResponse statuses
     * @throws Exception
     */
    public function testVariousStatusValuesCompliance(): void
    {
        $validStatuses = ['in-progress', 'completed', 'amended', 'entered-in-error', 'stopped'];

        foreach ($validStatuses as $status) {
            // Create QuestionnaireResponse with specific status - AI Generated status testing
            $responseUuid = $this->createTestQuestionnaireResponse($status);

            // Retrieve and verify
            $processingResult = $this->service->getOne($responseUuid);
            $this->assertTrue($processingResult->isValid(), "Status $status should be valid");

            $questionnaireResponse = $processingResult->getFirstDataResult();
            $this->assertInstanceOf(FHIRQuestionnaireResponse::class, $questionnaireResponse);
            $this->assertUSCoreCompliance($questionnaireResponse);

            // Verify status matches
            $this->assertEquals($status, $questionnaireResponse->getStatus()->getValue());
        }
    }

    /**
     * Test handling of QuestionnaireResponse items and nested items
     * @throws Exception
     */
    public function testQuestionnaireResponseItemsCompliance(): void
    {
        // Create QuestionnaireResponse with complex items - AI Generated complex items test
        $responseUuid = $this->createTestQuestionnaireResponseWithItems();

        $processingResult = $this->service->getOne($responseUuid);
        $this->assertTrue($processingResult->isValid(), "Complex items should be valid");
        $this->assertTrue($processingResult->hasData(), "Complex items result should be returned");
        $questionnaireResponse = $processingResult->getFirstDataResult();

        $this->assertUSCoreCompliance($questionnaireResponse);

        // Verify items are present and properly structured
        $items = $questionnaireResponse->getItem();
        $this->assertNotEmpty($items, "Items should be present");

        foreach ($items as $item) {
            $this->assertNotNull($item->getLinkId(), "Each item must have a linkId");
            $this->assertNotEmpty($item->getLinkId(), "LinkId must not be empty");

            // If item has answers, verify they're properly structured
            if (!empty($item->getAnswer())) {
                foreach ($item->getAnswer() as $answer) {
                    // Verify answer has a value (US Core supports valueString, valueDecimal, valueCoding)
                    $hasValue = $answer->getValueString() || $answer->getValueDecimal() || $answer->getValueCoding();
                    $this->assertTrue($hasValue, "Answer must have a value");
                }
            }
        }
    }

    /**
     * Test error handling with invalid data
     */
    public function testErrorHandlingWithInvalidData(): void
    {
        // Test with non-existent UUID - AI Generated error testing
        $nonExistentUuid = UuidRegistry::uuidToString(UuidRegistry::getRegistryForTable('questionnaire_response')->createUuid());

        $processingResult = $this->service->getOne($nonExistentUuid);
        $this->assertFalse($processingResult->isValid() && !empty($processingResult->getData()), "Non-existent UUID should return empty result");

        // Test parsing with corrupted JSON data
        $corruptedData = [
            'questionnaire_response_uuid' => $nonExistentUuid,
            'questionnaire_response' => '{"invalid": json}'
        ];

        $parsedResponse = $this->service->parseOpenEMRRecord($corruptedData);
        $this->assertEquals($nonExistentUuid, $parsedResponse->getId()->getValue());
    }

    // Helper methods for test setup and validation - AI Generated helper methods

    /**
     * Create a test patient for use in tests
     */
    private function createTestPatient(): void
    {
        $patientService = new PatientService();
        $result = $patientService->insert([
            'DOB' => '1990-01-01',
            'fname' => 'Integration',
            'lname' => 'Test Patient',
            'sex' => 'Female',
            'pubpid' => 'phpunit-integration-test-' . date("Y-m-d-H-i-s")
        ]);

        $this->assertTrue($result->isValid(), "Failed creating test patient");
        $this->testPatientData = $result->getFirstDataResult();
    }

    private function getQuestionnaireTemplate(): array
    {
        return [
            'resourceType' => 'Questionnaire',
            'id' => 'integration-test-questionnaire',
            'url' => 'http://example.com/questionnaire/integration-test',
            'version' => '1.0.0',
            'name' => 'IntegrationTestQuestionnaire',
            'title' => self::QUESTIONNAIRE_NAME,
            'status' => 'active',
            'item' => [
                [
                    'linkId' => '1',
                    'type' => 'string',
                    'text' => 'What is your name?'
                ],
                [
                    'linkId' => '2',
                    'type' => 'decimal',
                    'text' => 'What is your age?'
                ]
            ]
        ];
    }
    /**
     * Create a test questionnaire for use in tests
     * @throws Exception
     */
    private function createTestQuestionnaire(): void
    {
        $questionnaireService = new QuestionnaireService();

        // Load questionnaire template - AI Generated questionnaire template
        $questionnaireTemplate = $this->getQuestionnaireTemplate();

        $insertId = $questionnaireService->saveQuestionnaireResource($questionnaireTemplate);
        $this->assertNotEmpty($insertId, "Failed to create test questionnaire");

        // Get the UUID
        $binUuid = QueryUtils::fetchSingleValue('SELECT uuid FROM questionnaire_repository WHERE id = ?', 'uuid', [$insertId]);
        $this->testQuestionnaireUuid = UuidRegistry::uuidToString($binUuid);
    }

    /**
     * Create a test QuestionnaireResponse in the database
     * @param string $status Status for the response
     * @throws Exception
     * @return string UUID of created response
     */
    private function createTestQuestionnaireResponse(string $status = 'completed'): string
    {
        // Create QuestionnaireResponse JSON - AI Generated response structure
        $questionnaireResponseData = [
            'resourceType' => 'QuestionnaireResponse',
            'questionnaire' => 'http://example.com/questionnaire/integration-test|1.0.0',
            'status' => $status,
            'subject' => [
                'reference' => 'Patient/' . $this->testPatientData['uuid']
            ],
            'authored' => date('Y-m-d\TH:i:s\Z'),
            'author' => [
                'reference' => 'Practitioner/test-practitioner'
            ],
            'item' => [
                [
                    'linkId' => '1',
                    'text' => 'What is your name?',
                    'answer' => [
                        [
                            'valueString' => 'Integration Test'
                        ]
                    ]
                ]
            ]
        ];

        // Insert into database
        $questionnaireResponseService = new QuestionnaireResponseService();
        $questionnaireTemplate = $this->getQuestionnaireTemplate();
        $questionnaireTemplate['id'] = $this->testQuestionnaireUuid;
        $result = $questionnaireResponseService->saveQuestionnaireResponse(
            json_encode($questionnaireResponseData),
            $this->testPatientData['pid'],
            null,
            null,
            null,
            json_encode($questionnaireTemplate)
        );
        $this->assertIsArray($result, "Failed to create test QuestionnaireResponse");
        $this->assertNotNull($result['response_id']);

        return $result['response_id'];
    }

    /**
     * Create a test QuestionnaireResponse with complex items
     * @throws Exception
     * @return string UUID of created response
     */
    private function createTestQuestionnaireResponseWithItems(): string
    {
        // Create complex QuestionnaireResponse with nested items - AI Generated complex structure
        $questionnaireResponseData = [
            'resourceType' => 'QuestionnaireResponse',
            'questionnaire' => 'http://example.com/questionnaire/integration-test|1.0.0',
            'status' => 'completed',
            'subject' => [
                'reference' => 'Patient/' . $this->testPatientData['uuid']
            ],
            'authored' => date('Y-m-d\TH:i:s\Z'),
            'item' => [
                [
                    'linkId' => '1',
                    'text' => 'Personal Information',
                    'item' => [
                        [
                            'linkId' => '1.1',
                            'text' => 'First Name',
                            'answer' => [
                                [
                                    'valueString' => 'John'
                                ]
                            ]
                        ],
                        [
                            'linkId' => '1.2',
                            'text' => 'Age',
                            'answer' => [
                                [
                                    'valueDecimal' => 35.5
                                ]
                            ]
                        ]
                    ]
                ],
                [
                    'linkId' => '2',
                    'text' => 'Assessment',
                    'answer' => [
                        [
                            'valueCoding' => [
                                'system' => 'http://loinc.org',
                                'code' => '72133-2',
                                'display' => 'Anxiety severity'
                            ]
                        ]
                    ]
                ]
            ]
        ];

        // Save complex response to database - AI Generated complex save
        $questionnaireResponseService = new QuestionnaireResponseService();

        $questionnaireTemplate = $this->getQuestionnaireTemplate();
        $questionnaireTemplate['id'] = $this->testQuestionnaireUuid;
        $result = $questionnaireResponseService->saveQuestionnaireResponse(
            json_encode($questionnaireResponseData),
            $this->testPatientData['pid'],
            null,
            null,
            null,
            json_encode($questionnaireTemplate)
        );
        $this->assertNotEmpty($result, "Failed to create complex test QuestionnaireResponse");
        $this->assertNotNull($result['response_id']);

        return $result['response_id'];
    }

    /**
     * Clean up test data from database
     */
    private function cleanupTestData(): void
    {
        // Clean up questionnaire responses - AI Generated cleanup
        QueryUtils::sqlStatementThrowException(
            "DELETE FROM questionnaire_response WHERE questionnaire_foreign_id IN (SELECT id FROM questionnaire_repository WHERE name = ? OR name = ?)",
            [self::QUESTIONNAIRE_NAME, self::QUESTIONNAIRE_NAME_2]
        );

        // Clean up questionnaires
        QueryUtils::sqlStatementThrowException(
            "DELETE FROM uuid_registry WHERE uuid IN (SELECT uuid FROM questionnaire_repository WHERE name = ? OR name = ?)",
            [self::QUESTIONNAIRE_NAME, self::QUESTIONNAIRE_NAME_2]
        );
        QueryUtils::sqlStatementThrowException(
            "DELETE FROM questionnaire_repository WHERE name = ? OR name = ?",
            [self::QUESTIONNAIRE_NAME, self::QUESTIONNAIRE_NAME_2]
        );

        // Clean up test patient
        if (!empty($this->testPatientData['pid'])) {
            QueryUtils::sqlStatementThrowException(
                "DELETE FROM patient_data WHERE pid = ?",
                [$this->testPatientData['pid']]
            );
        }
    }

    /**
     * Assert that a QuestionnaireResponse meets US Core 8.0 compliance requirements
     * @param FHIRQuestionnaireResponse $response
     */
    private function assertUSCoreCompliance(FHIRQuestionnaireResponse $response): void
    {
        // Required elements (mustSupport = true, min = 1)

        // questionnaire (1..1) - AI Generated compliance check
        $this->assertNotNull($response->getQuestionnaire(), "US Core requires questionnaire element (1..1)");
        $this->assertNotEmpty($response->getQuestionnaire()->getValue(), "Questionnaire canonical URL must not be empty");

        // status (1..1) - AI Generated status validation
        $this->assertNotNull($response->getStatus(), "US Core requires status element (1..1)");
        $validStatuses = ['in-progress', 'completed', 'amended', 'entered-in-error', 'stopped'];
        $this->assertContains($response->getStatus()->getValue(), $validStatuses, "Status must be US Core compliant");

        // subject (1..1) - AI Generated subject validation
        $this->assertNotNull($response->getSubject(), "US Core requires subject element (1..1)");
        $this->assertNotEmpty($response->getSubject()->getReference(), "Subject reference must not be empty");

        // authored (1..1) - AI Generated authored validation
        $this->assertNotNull($response->getAuthored(), "US Core requires authored element (1..1)");
        $this->assertNotEmpty($response->getAuthored()->getValue(), "Authored datetime must not be empty");

        // Optional but supported elements

        // author (0..1) - if present, must be valid reference - AI Generated optional validation
        if (!empty($response->getAuthor())) {
            $this->assertNotEmpty($response->getAuthor()->getReference(), "Author reference must not be empty if present");
            $authorRef = $response->getAuthor()->getReference();
            $supportedTypes = ['Practitioner', 'Organization', 'Patient', 'PractitionerRole', 'Device', 'RelatedPerson'];
            $matchesType = false;
            foreach ($supportedTypes as $type) {
                if (str_starts_with($authorRef, $type . '/')) {
                    $matchesType = true;
                    break;
                }
            }
            $this->assertTrue($matchesType, "Author must reference a US Core supported resource type");
        }

        // item (0..*) - if present, must have required elements - AI Generated item validation
        $items = $response->getItem();
        if (!empty($items)) {
            foreach ($items as $item) {
                $this->assertNotNull($item->getLinkId(), "Each item must have a linkId (1..1)");
                $this->assertNotEmpty($item->getLinkId(), "LinkId must not be empty");

                // If answers are present, they should be valid
                if (!empty($item->getAnswer())) {
                    foreach ($item->getAnswer() as $answer) {
                        // US Core supports valueString, valueDecimal, valueCoding
                        $hasValidValue = $answer->getValueString() || $answer->getValueDecimal() || $answer->getValueCoding();
                        // Note: We're being lenient here as other value types might be supported
                        $this->assertTrue($hasValidValue, "Answer must have a valid value (1..1)");
                    }
                }
            }
        }
    }
    // end AI Generated sections by Claude.AI
}
