<?php

/*
 * FhirQuestionnaireRestControllerTest.php
 * @package openemr
 * @link      http://www.open-emr.org
 * @author    Stephen Nielson <snielson@discoverandchange.com>
 * @copyright Copyright (c) 2025 Stephen Nielson <snielson@discoverandchange.com>
 * @license   https://github.com/openemr/openemr/blob/master/LICENSE GNU General Public License 3
 */

namespace OpenEMR\Tests\RestControllers\FHIR;

use OpenEMR\Common\Database\QueryUtils;
use OpenEMR\Common\Http\HttpRestRequest;
use OpenEMR\Common\Logging\SystemLogger;
use OpenEMR\Common\Uuid\UuidRegistry;
use OpenEMR\RestControllers\FHIR\FhirQuestionnaireRestController;
use OpenEMR\Services\FHIR\FhirQuestionnaireService;
use OpenEMR\Services\FHIR\Questionnaire\FhirQuestionnaireFormService;
use OpenEMR\Services\ObservationService;
use OpenEMR\Services\QuestionnaireService;
use PHPUnit\Framework\TestCase;
use Symfony\Component\HttpFoundation\Response;
use PHPUnit\Framework\MockObject\Exception as MockException;
use Exception;

class FhirQuestionnaireRestControllerIntegrationTest extends TestCase
{
    const QUESTIONNAIRE_NAME_1 = 'PHPUnit Test Questionnaire';
    const QUESTIONNAIRE_NAME_2 = 'blah blah blah';


    /**
     * @var array Saved off session storage while we run this test.
     */
    private array $originalSession;

    private readonly string $questionnaireTemplate;

    protected function setUp(): void
    {
        parent::setUp(); // TODO: Change the autogenerated stub
        // TODO: When we have a session storage we can mock we want to get away from
        // this session behavior
        $this->originalSession = $_SESSION;
        $this->questionnaireTemplate = file_get_contents(__DIR__ . '/../../data/Services/FHIR/Questionnaire/questionnaire-sdc-pathology.json');
    }

    protected function tearDown(): void
    {
        parent::tearDown();
        $_SESSION = $this->originalSession; // restore the session sadly.
        // need to clean up the database records
        $name = self::QUESTIONNAIRE_NAME_1;
        $name2 = self::QUESTIONNAIRE_NAME_2;
        QueryUtils::sqlStatementThrowException("DELETE FROM uuid_registry WHERE uuid IN (select uuid FROM questionnaire_repository WHERE name =? OR name = ?)", [$name, $name2]);
        QueryUtils::sqlStatementThrowException("DELETE FROM questionnaire_repository WHERE name = ? OR name = ?", [$name, $name2]);
    }

    /**
     * @return void
     * @throws MockException|Exception
     */
    public function testOneReturnsQuestionnaireResponseForUserContext(): void
    {
        $_SESSION['authUserID'] = QueryUtils::fetchSingleValue('select id FROM users ORDER BY id LIMIT 1', 'id');
        // save the questionnaire we want returned and verify the system returns the questionnaire
        $uuid = $this->createQuestionnaireWithName(self::QUESTIONNAIRE_NAME_1);

        // test that an inserted questionnaire form will retrieve the questionnaire via the service
        $formService = new FhirQuestionnaireFormService();
        $questionnaireService = new FhirQuestionnaireService();
        $questionnaireService->addMappedService($formService);
        $controller = new FhirQuestionnaireRestController(new SystemLogger(), $questionnaireService);
        $request = $this->createMock(HttpRestRequest::class);
        $response = $controller->one($request, $uuid);

        $this->assertEquals(Response::HTTP_OK, $response->getStatusCode(), "Expected valid response");
        $response->getBody()->rewind();
        $content = $response->getBody()->getContents();
        $this->assertNotEmpty($content, "Response content should be returned");
        $jsonObj = json_decode($content, true);
        $this->assertNotFalse($jsonObj, "Json response should be decodeable");

        $this->assertEquals($uuid, $jsonObj['id'], "Should return the proper questionnaire id");
        $this->assertEquals('4.0.0', $jsonObj['version']);
        // save mechanism lowers everything and removes spaces....
        $this->assertEquals('phpunittestquestionnaire', $jsonObj['name']);
        $this->assertCount(2, $jsonObj['item']);
        $this->assertEquals("1", $jsonObj['item'][0]['linkId']);
        $this->assertEquals("2", $jsonObj['item'][1]['linkId']);
    }

    /**
     * @return void
     * @throws MockException|Exception
     */
    public function testListReturnsAllResults(): void
    {
        $_SESSION['authUserID'] = QueryUtils::fetchSingleValue('select id FROM users ORDER BY id LIMIT 1', 'id');
        // save the questionnaire we want returned and verify the system returns the questionnaire

        $uuid = $this->createQuestionnaireWithName(self::QUESTIONNAIRE_NAME_1);
        // create a second one that should be ignored in the search
        $uuid2 = $this->createQuestionnaireWithName(self::QUESTIONNAIRE_NAME_2);
        $observationService = $this->createPartialMock(QuestionnaireService::class, ['search']);
        $observationService->method('search')
            ->willReturn([

            ]);
        $formService = new FhirQuestionnaireFormService();
        $questionnaireService = new FhirQuestionnaireService();
        $questionnaireService->addMappedService($formService);
        $controller = new FhirQuestionnaireRestController(new SystemLogger(), $questionnaireService);
        $request = $this->createMock(HttpRestRequest::class);
        $request->method('getQueryParams')->willReturn([]);
        $response = $controller->list($request);

        $this->assertEquals(Response::HTTP_OK, $response->getStatusCode(), "Expected valid response");
        $response->getBody()->rewind();
        $content = $response->getBody()->getContents();
        $this->assertNotEmpty($content, "Response content should be returned");
        $jsonObj = json_decode($content, true);
        $this->assertNotFalse($jsonObj, "Json response should be decodeable");
        $this->assertEquals("Bundle", $jsonObj['resourceType']);
        $this->assertCount(2, $jsonObj['entry']);
        $this->assertArrayHasKey("resource", $jsonObj['entry'][0]);
        $this->assertEquals($uuid, $jsonObj['entry'][0]['resource']['id'], "Should return the proper questionnaire id");
        $this->assertEquals($uuid2, $jsonObj['entry'][1]['resource']['id'], "Should return the proper questionnaire id");
    }

    /**
     * @return void
     * @throws MockException|Exception
     */
    public function testListWithSearchOnlyReturnsOneResult(): void
    {
        $_SESSION['authUserID'] = QueryUtils::fetchSingleValue('select id FROM users ORDER BY id LIMIT 1', 'id');
        // save the questionnaire we want returned and verify the system returns the questionnaire

        $uuid = $this->createQuestionnaireWithName(self::QUESTIONNAIRE_NAME_1);
        // create a second one that should be ignored in the search
        $this->createQuestionnaireWithName(self::QUESTIONNAIRE_NAME_2);

        // test that an inserted questionnaire form will retrieve the questionnaire via the service
        $formService = new FhirQuestionnaireFormService();
        $questionnaireService = new FhirQuestionnaireService();
        $questionnaireService->addMappedService($formService);
        $controller = new FhirQuestionnaireRestController(new SystemLogger(), $questionnaireService);
        $request = $this->createMock(HttpRestRequest::class);
        // modifier
        $request->method('getQueryParams')->willReturn([
            'title:eq' => self::QUESTIONNAIRE_NAME_1
        ]);
        $response = $controller->list($request);

        $this->assertEquals(Response::HTTP_OK, $response->getStatusCode(), "Expected valid response");
        $response->getBody()->rewind();
        $content = $response->getBody()->getContents();
        $this->assertNotEmpty($content, "Response content should be returned");
        $jsonObj = json_decode($content, true);
        $this->assertNotFalse($jsonObj, "Json response should be decodeable");
//        var_dump($jsonObj);

        $this->assertEquals("Bundle", $jsonObj['resourceType']);
        $this->assertCount(1, $jsonObj['entry']);
        $this->assertArrayHasKey("resource", $jsonObj['entry'][0]);
        $jsonObj = $jsonObj['entry'][0]['resource'];

        $this->assertEquals($uuid, $jsonObj['id'], "Should return the proper questionnaire id");
        $this->assertEquals('4.0.0', $jsonObj['version']);
        // save mechanism lowers everything and removes spaces....
        $this->assertEquals('phpunittestquestionnaire', $jsonObj['name']);
        $this->assertCount(2, $jsonObj['item']);
        $this->assertEquals("1", $jsonObj['item'][0]['linkId']);
        $this->assertEquals("2", $jsonObj['item'][1]['linkId']);
    }

    /**
     * @param string $name
     * @return string
     * @throws Exception
     */
    private function createQuestionnaireWithName(string $name): string
    {
        $questionnaireContents = $this->questionnaireTemplate;
        $questionnaireAsJson = json_decode($questionnaireContents, true);
        $questionnaireAsJson['name'] = $questionnaireAsJson['title'] = $name;
        $questionnaireService = new QuestionnaireService();
        $insertId = $questionnaireService->saveQuestionnaireResource($questionnaireAsJson);
        // need to grab the record because we don't get ANY kind of uuid from this
        $binUuid = QueryUtils::fetchSingleValue('SELECT uuid FROM questionnaire_repository WHERE id = ?', 'uuid', [$insertId]);
        $this->assertNotEmpty($binUuid, "Questionnaire uuid resource should be returned");
        return UuidRegistry::uuidToString($binUuid);
    }
}
