<?php

/*
 * AuthorizationGrantFlowTest.php
 * @package openemr
 * @link      http://www.open-emr.org
 * @author    Stephen Nielson <snielson@discoverandchange.com>
 * @copyright Copyright (c) 2025 Stephen Nielson <snielson@discoverandchange.com>
 * @license   https://github.com/openemr/openemr/blob/master/LICENSE GNU General Public License 3
 */

namespace OpenEMR\Tests\Api;

use Monolog\Level;
use OpenEMR\Common\Auth\OpenIDConnect\Entities\ClientEntity;
use OpenEMR\Common\Auth\OpenIDConnect\Entities\ServerScopeListEntity;
use OpenEMR\Common\Auth\OpenIDConnect\Repositories\ClientRepository;
use OpenEMR\Common\Csrf\CsrfUtils;
use OpenEMR\Common\Http\HttpRestRequest;
use OpenEMR\Common\Logging\SystemLogger;
use OpenEMR\Core\Kernel;
use OpenEMR\Core\OEHttpKernel;
use OpenEMR\RestControllers\AuthorizationController;
use OpenEMR\RestControllers\SMART\SMARTAuthorizationController;
use OpenEMR\Services\TrustedUserService;
use OpenEMR\Services\UserService;
use PHPUnit\Framework\TestCase;
use Symfony\Component\EventDispatcher\EventDispatcher;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\Session\Storage\MockFileSessionStorage;
use Symfony\Component\HttpKernel\Controller\ControllerResolverInterface;
use Symfony\Component\HttpFoundation\Session\Session;

class AuthorizationGrantFlowTest extends TestCase
{
    private array $clientsToCleanup = [];

    protected function tearDown(): void
    {
        parent::tearDown(); // TODO: Change the autogenerated stub
        $clientRepository = new ClientRepository();
        foreach ($this->clientsToCleanup as $clientIdentifier) {
            $clientEntity = new ClientEntity();
            $clientEntity->setIdentifier($clientIdentifier);
            $clientRepository->remove($clientEntity, true);
        }
    }

    private function getMockSession(): Session
    {
        $session = new Session(new MockFileSessionStorage());
        $session->start();
        $session->set("site_id", "default");
        return $session;
    }

    /**
     * @return void
     * @throws \League\OAuth2\Server\Exception\OAuthServerException
     * @throws \PHPUnit\Framework\MockObject\Exception
     */
    public function testSuccessfulAuthorizationGrantFlowWithLoginFlow(): void
    {
        $serverScopes = new ServerScopeListEntity();
        $scopes = $serverScopes->getAllSupportedScopesList();
        $scopesString = implode(" ", $scopes);
//        $scopesString = ApiTestClient::ALL_SCOPES;
        $redirectUri = "http://localhost:8080/oauth2/callback";
        $dispatcher = new EventDispatcher();
        $controller = $this->createMock(ControllerResolverInterface::class);
        $kernel = new OEHttpKernel($dispatcher, $controller);
        // set the globals.php kernel which doesn't do much... need to reconicle these two
        // set our site_addr_oath so we use it properly
        $kernel->getGlobalsBag()->set('site_addr_oath', $this->getBaseUrlApi());
        $kernel->getGlobalsBag()->set('kernel', new Kernel($dispatcher));
        [$clientIdentifier, $clientSecret] = $this->requestTestRegistrationEndpoint($kernel, $redirectUri, $scopesString);

        // now test the authorization flow
        $session = $this->getMockSession();
        $session->set('nonce', 'test_nonce');
        $loginLocation = $this->requestTestAuthorizeEndpoint($session, $scopesString, $clientIdentifier, $kernel);

        // now attempt the login flow, redirecting to the blank login screen
        $patientSelectPage = $this->requestTestLoginFlow($loginLocation, $session, $kernel);
//        $this->assertStringEndsWith(AuthorizationController::ENDPOINT_SCOPE_AUTHORIZE_CONFIRM, $scopeConfirmationPage, 'Redirect location should end with scope authorize confirm endpoint');
        $this->assertStringEndsWith(SMARTAuthorizationController::PATIENT_SELECT_PATH, $patientSelectPage, 'Redirect location should end with SMART on FHIR patient select path');

        // now do the patient selection flow
        $scopeConfirmationPage = $this->requestTestPatientSelectionFlow($patientSelectPage, $session, $kernel);

        // now for the scope confirmation page
        $this->requestTestScopeConfirmationPageEndpoint($session, $scopeConfirmationPage, $kernel);

        // now we will simulate the user confirming the scopes by posting to the /auth/code endpoint
        $location = $this->requestTestAuthorizationCodeEndpoint($session, $scopesString, $kernel, $redirectUri);

        // next step is to grab the code from the redirect URI and exchange it for an access token
        $this->requestTestTokenEndpoint(
            $location,
            $redirectUri,
            $clientIdentifier,
            $clientSecret,
            $session,
            $scopesString,
            $kernel
        );
    }

    /**
     * @param string $redirectUri
     * @param string $scopesString
     * @return array
     */
    private function requestTestRegistrationEndpoint(OEHttpKernel $kernel, string $redirectUri, string $scopesString): array
    {
        $jsonRequest = [
            "application_type" => "private",
            "redirect_uris" => [
                $redirectUri
            ],
            "client_name" => "Test Client",
            "token_endpoint_auth_method" => "client_secret_post",
            "contacts" => ["test@open-emr.org"],
            "scope" => $scopesString
        ];

        $session = $this->getMockSession();
        $originalSessionId = $session->getId();
        $authController = $this->getAuthorizationController($session, $kernel);
        $registrationRequest = HttpRestRequest::create("/oauth2/default/register", "POST", [], [], [], [
            'CONTENT_TYPE' => 'application/json',
        ], json_encode($jsonRequest));
        $response = $authController->clientRegistration($registrationRequest);
        $this->assertEquals(200, $response->getStatusCode(), " Request to register client should have been successful");

        $contents = $response->getBody()->getContents();
        $this->assertNotEmpty($contents, 'Response body should not be empty');
        $json = json_decode($contents, true);
        $this->assertArrayHasKey('client_id', $json, 'Response should contain client_id');
        $clientIdentifier = $json['client_id'];
        $this->clientsToCleanup[] = $clientIdentifier;
        $this->assertArrayHasKey("client_secret", $json, 'Response should contain client_secret');
        $this->assertArrayHasKey("client_name", $json, 'Response should contain client_name');
        $clientSecret = $json['client_secret'];

        $this->assertEquals($scopesString, $json['scope'], 'Response should contain the correct scopes');

        $this->assertNotEquals($originalSessionId, $session->getId(), "Session ID should have changed after registration due to invalidation");

        // enable the client
        $clientRepository = new ClientRepository();
        $clientEntity = $clientRepository->getClientEntity($clientIdentifier);
        $clientRepository->saveIsEnabled($clientEntity, true);
        return [$clientIdentifier, $clientSecret];
    }

    /**
     * @param string $scopesString
     * @param mixed $clientIdentifier
     * @param OEHttpKernel $kernel
     * @return string
     */
    private function requestTestAuthorizeEndpoint(Session $session, string $scopesString, mixed $clientIdentifier, OEHttpKernel $kernel): string
    {
        $request = HttpRestRequest::create("/oauth2/default/authorize", "GET", [
            "scope" => $scopesString,
            "client_id" => $clientIdentifier,
            "response_type" => "code",
            "redirect_uri" => "http://localhost:8080/oauth2/callback",
            "state" => "test_state",
        ]);
        $originalSessionId = $session->getId();
        $authController = new AuthorizationController($session, $kernel, true);
        $response = $authController->oauthAuthorizationFlow($request);
        $this->assertEquals(Response::HTTP_TEMPORARY_REDIRECT, $response->getStatusCode(), 'Authorization request should redirect');
        $this->assertNotEmpty($response->getHeader('Location'), 'Response header should not be empty');
        $location = $response->getHeader('Location')[0];
        $this->assertStringContainsString('/provider/login', $location, 'Redirect location should contain provider login when skip authorization is not set');
        return $location;
    }

    /**
     * @param array|null $loginLocation
     * @param Session $session
     * @param OEHttpKernel $kernel
     * @return string
     */
    private function requestTestLoginFlow(string $loginLocation, Session $session, OEHttpKernel $kernel): string
    {
        $request = HttpRestRequest::create($loginLocation);
        CsrfUtils::setupCsrfKey($session);
        $authController = new AuthorizationController($session, $kernel, true);
        $loginScreenResponse = $authController->userLogin($request);

        $this->assertEquals(Response::HTTP_OK, $loginScreenResponse->getStatusCode(), 'Login screen should return 200 status code');

        // now we will simulate a user login
        $request = HttpRestRequest::create($loginLocation, "POST", [
            "username" => "admin",
            "password" => "pass",
            // this is not to be confused with the other 'role' variables
            "user_role" => "api", // valid options are "api" or "portal-api"
            "csrf_token_form" => CsrfUtils::collectCsrfToken('oauth2', $session)
        ], [], []);
        // we have to do overrideGlobals since AuthUtils will not work with the HttpRestRequest but instead requires superglobals
        $request->overrideGlobals();
        $authController = new AuthorizationController($session, $kernel, true);
        $loginPostResponse = $authController->userLogin($request);
        $this->assertNotEmpty($loginPostResponse->getHeader('Location'), 'Login POST response should have a Location header');
        $this->assertEquals(Response::HTTP_TEMPORARY_REDIRECT, $loginPostResponse->getStatusCode(), 'Login POST should redirect');
        $this->assertFalse($request->request->has("username"), "Request should have username removed");
        $this->assertFalse($request->request->has("password"), "Request should have password removed");
        $this->assertEquals(0, $session->get("persist_login"), "Session should not have persist_login set");
        $userService = new UserService();
        $user = $userService->getUserByUsername("admin");
        $this->assertEquals([
            "name" => $user['fname'] . ' ' . $user['lname'],
            "family_name" => $user['lname'],
            "given_name" => "",
            "preferred_username" => $user['username'],
            "middle_name" => null,
            "profile" => "",
            "picture" => "",
            "website" => "",
            "gender" => "",
            "birthdate" => "",
            "zoneinfo" => "",
            "locale" => "US",
            "updated_at" => "",
            "email" => null,
            "email_verified" => true,
            "phone_number" => null,
            "phone_number_verified" => true,
            "address" => "  ",
            "zip" => null,
            "fhirUser" => $this->getBaseUrlApi() . "/apis/default/fhir/Person/" . $user['uuid'],
            'nickname' => '',
            'api:fhir' => true,
            'api:oemr' => true,
            'api:port' => true,
        ], $session->get('claims'));
        return $loginPostResponse->getHeader('Location')[0];
    }

    private function getAuthorizationController(Session $session, OEHttpKernel $kernel): AuthorizationController
    {
        $authController = new AuthorizationController($session, $kernel, true);
        $authController->setSystemLogger(new SystemLogger(Level::Error));
        return $authController;
    }

    /**
     * @param Session $session
     * @param string $scopeConfirmationPage
     * @param OEHttpKernel $kernel
     * @return array
     */
    private function requestTestScopeConfirmationPageEndpoint(Session $session, string $scopeConfirmationPage, OEHttpKernel $kernel): void
    {
        $originalSessionId = $session->getId();
        $request = HttpRestRequest::create($scopeConfirmationPage);
        $authController = new AuthorizationController($session, $kernel, true);
        $scopeConfirmationResponse = $authController->scopeAuthorizeConfirm($request);
        $this->assertEquals(Response::HTTP_OK, $scopeConfirmationResponse->getStatusCode(), 'Scope confirmation page should return 200 status code');
    }

    /**
     * @param Session $session
     * @param string $scopesString
     * @param OEHttpKernel $kernel
     * @param string $redirectUri
     * @return string
     */
    private function requestTestAuthorizationCodeEndpoint(
        Session $session,
        string $scopesString,
        OEHttpKernel $kernel,
        string $redirectUri
    ): string {
        $scope = explode(" ", $scopesString);
        $scopeArray = array_combine($scope, $scope);
        $originalSessionId = $session->getId();
        $request = HttpRestRequest::create("/oauth2/default" . AuthorizationController::DEVICE_CODE_ENDPOINT, "POST", [
            "csrf_token_form" => CsrfUtils::collectCsrfToken('oauth2', $session),
            // this is an array of scopes [scopeIdentifier] => scopeIdentifier
            "scope" => $scopeArray,
            'proceed' => "1",
        ]);
        $authController = new AuthorizationController($session, $kernel, true);
        $deviceCodeResponse = $authController->authorizeUser($request);
        $this->assertEquals(Response::HTTP_FOUND, $deviceCodeResponse->getStatusCode(), 'Device code response should return 302 status code');
        $this->assertNotEmpty($deviceCodeResponse->getHeader('Location'), 'Response header should not be empty');
        $location = $deviceCodeResponse->getHeader('Location')[0];
        $this->assertStringStartsWith($redirectUri, $location, 'Redirect location should start with the redirect URI');
        $this->assertStringContainsString('code=', $location, 'Redirect location should contain code parameter');
        $this->assertStringContainsString('state=test_state', $location, 'Redirect location should contain state parameter');
        $this->assertNotEquals($originalSessionId, $session->getId(), "Session ID should have changed after authorization due to invalidation");
        return $location;
    }

    /**
     * @param string $location
     * @param string $redirectUri
     * @param string $clientIdentifier
     * @param string $clientSecret
     * @param Session $session
     * @param string $scopesString
     * @param OEHttpKernel $kernel
     * @return void
     */
    private function requestTestTokenEndpoint(
        string $location,
        string $redirectUri,
        string $clientIdentifier,
        string $clientSecret,
        Session $session,
        string $scopesString,
        OEHttpKernel $kernel
    ): void {
        $redirectUrlParts = parse_url($location);
        parse_str($redirectUrlParts['query'], $queryParams);
        $this->assertArrayHasKey('code', $queryParams, 'Redirect URL should contain code parameter');
        $code = $queryParams['code'];
        $this->assertNotEmpty($code, 'Code should not be empty');
        $this->assertArrayHasKey('state', $queryParams, 'Redirect URL should contain state parameter');
        $this->assertEquals('test_state', $queryParams['state'], 'State parameter should match the original state');


        // now verify the code was saved to the database
        $trustedUserService = new TrustedUserService();
        $trustedUser = $trustedUserService->getTrustedUserByCode($code);
        $this->assertNotEmpty($trustedUser, 'Trusted user should not be empty for generated code');
        $this->assertNotEmpty($trustedUser['session_cache'], 'Trusted user session cache should not be empty for generated code');

        // once we have the access token, we want to verify we also got the id token and the refresh token
        $tokenRequest = HttpRestRequest::create("/oauth2/default/token", "POST", [
            "grant_type" => "authorization_code",
            "code" => $code,
            "redirect_uri" => $redirectUri,
        ], [], [], [
            'HTTP_AUTHORIZATION' => 'Basic ' . base64_encode($clientIdentifier . ':' . $clientSecret),
        ]);
        $session = $this->getMockSession();
        $originalSessionId = $session->getId();
        $authController = $this->getAuthorizationController($session, $kernel);
        $tokenResponse = $authController->oauthAuthorizeToken($tokenRequest);
        $this->assertEquals(Response::HTTP_OK, $tokenResponse->getStatusCode(), 'Token request should return 200 status code');
        $contents = $tokenResponse->getBody()->__toString();
        $this->assertNotEmpty($contents, 'Token response body should not be empty');
        $json = json_decode($contents, true);
        $this->assertArrayHasKey('access_token', $json, 'Token response should contain access_token');
        $this->assertArrayHasKey('id_token', $json, 'Token response should contain id_token');
        $this->assertArrayHasKey('refresh_token', $json, 'Token response should contain refresh_token');
        $this->assertNotEquals($originalSessionId, $session->getId(), "Session ID should be a new session");
    }

    private function requestTestPatientSelectionFlow(string $patientSelectPage, Session $session, OEHttpKernel $kernel)
    {
        // we will grab the patient selection page and then simulate the user selecting a patient
        $request = HttpRestRequest::create($patientSelectPage);
        $authController = $this->getAuthorizationController($session, $kernel);
        $patientSelectionResponse = $authController->dispatchSMARTAuthorizationEndpoint($patientSelectPage, $request);
        $this->assertEquals(Response::HTTP_OK, $patientSelectionResponse->getStatusCode(), 'Patient selection page should return 200 status code');
        //$this->assertStringContainsString(SMARTAuthorizationController::PATIENT_SELECT_PATH, $patientSelectionResponse->getContent(), 'Patient selection page should contain patient select path');

        // now we will make a post request to select a patient
        $patientSelectionSubmissionPage = $authController->authBaseFullUrl . SMARTAuthorizationController::PATIENT_SELECT_CONFIRM_ENDPOINT;
        $request = HttpRestRequest::create($patientSelectionSubmissionPage, "POST", [
            "csrf_token" => CsrfUtils::collectCsrfToken('oauth2', $session),
            "patient_id" => "1", // assuming patient with ID 1 exists
            "proceed" => "1",
        ]);
        $authController = $this->getAuthorizationController($session, $kernel);
        $patientSelectionSubmissionResponse = $authController->dispatchSMARTAuthorizationEndpoint($patientSelectionSubmissionPage, $request);
        $this->assertEquals(Response::HTTP_TEMPORARY_REDIRECT, $patientSelectionSubmissionResponse->getStatusCode(), 'Patient selection submission should redirect');
        $scopeConfirmationPage = $patientSelectionSubmissionResponse->getHeader('Location')[0];
        $this->assertStringEndsWith(AuthorizationController::ENDPOINT_SCOPE_AUTHORIZE_CONFIRM, $scopeConfirmationPage, 'Redirect location should end with scope authorize confirm endpoint');
        return $scopeConfirmationPage;
    }

    /**
     * @return string
     */
    private function getBaseUrlApi(): string
    {
        $url = getenv("OPENEMR_BASE_URL_API", true);
        if (is_array($url)) {
            return (string)$url[0];
        } else if (is_string($url)) {
            return $url;
        } else {
            return "https://localhost:9300";
        }
    }
}
