<?php

/**
 * FHIR Resource Service class example for implementing the methods that are typically used with FHIR resources via the
 * api.
 *
 * @package   OpenEMR
 * @link      http://www.open-emr.org
 *
 * @author    Stephen Nielson <stephen@nielson.org>
 * @copyright Copyright (c) 2022 Stephen Nielson <stephen@nielson.org>
 * @license   https://github.com/openemr/openemr/blob/master/LICENSE GNU General Public License 3
 */

namespace OpenEMR\Services\FHIR;

use OpenEMR\FHIR\R4\FHIRResource\FHIRDomainResource;
use OpenEMR\Services\FHIR\QuestionnaireResponse\FhirQuestionnaireResponseFormService;
use OpenEMR\Services\FHIR\Traits\FhirServiceBaseEmptyTrait;
use OpenEMR\Services\FHIR\Traits\MappedServiceTrait;
use OpenEMR\Services\FHIR\Traits\PatientSearchTrait;
use OpenEMR\Services\FHIR\Traits\VersionedProfileTrait;
use OpenEMR\Services\Search\FhirSearchParameterDefinition;
use OpenEMR\Services\Search\SearchFieldType;
use OpenEMR\Services\Search\ServiceField;
use OpenEMR\Validators\ProcessingResult;
use Symfony\Component\EventDispatcher\EventDispatcher;
use Symfony\Component\EventDispatcher\GenericEvent;
use InvalidArgumentException;

class FhirQuestionnaireResponseService extends FhirServiceBase implements
    IResourceReadableService,
    IResourceSearchableService,
    IResourceCreatableService,
    IResourceUSCIGProfileService
{
    /**
     * If you'd prefer to keep out the empty methods that are doing nothing uncomment the following helper trait
     */
    use FhirServiceBaseEmptyTrait;
    use PatientSearchTrait;
    use MappedServiceTrait;
    use VersionedProfileTrait;

    const USCGI_PROFILE_URI = 'http://hl7.org/fhir/us/core/StructureDefinition/us-core-questionnaireresponse';

    private EventDispatcher $dispatcher;


    public function __construct(?string $fhirApiURL = null)
    {
        parent::__construct($fhirApiURL);
    }

    public function setEventDispatcher(EventDispatcher $dispatcher)
    {
        $this->dispatcher = $dispatcher;
    }

    public function getEventDispatcher()
    {
        if (!isset($this->dispatcher)) {
            $this->dispatcher = new EventDispatcher();
        }
        return $this->dispatcher;
    }

    public function insert(FHIRDomainResource $fhirResource): ProcessingResult
    {
        if ($fhirResource->get_fhirElementName() != 'QuestionnaireResponse') {
            throw new InvalidArgumentException("Invalid resource type for this repository");
        }

        // TODO: @adunsulag if we bring this into core we need to refactor out the generic event
        // TODO: @adunsulag we originally had events then needed to switch to the mapped service for searching, do we drop the event?
        $event = $this->getEventDispatcher()->dispatch(new GenericEvent($fhirResource), 'fhir.questionnaire_response.pre_insert');
        if ($event->isPropagationStopped()) {
            return $event->getArgument('result');
        } else {
            return parent::insert($fhirResource); // TODO: Change the autogenerated stub
        }
    }

    public function getAll($fhirSearchParameters, $puuidBind = null): ProcessingResult
    {
        $event = $this->getEventDispatcher()->dispatch(new GenericEvent($fhirSearchParameters), 'fhir.questionnaire_response.search');
        if ($event->hasArgument('result')) {
            $result = $event->getArgument('result');
            if ($result instanceof ProcessingResult) {
                if ($event->isPropagationStopped() || !$result->isValid()) {
                    return $result;
                }
            } else {
                $result->setInternalErrors(xlt("The event should have returned a ProcessingResult object"));
            }
            if ($result->isValid()) {
                $processingResult = $this->searchAllServicesWithSupportedFields($fhirSearchParameters, $puuidBind);
                $result->addProcessingResult($processingResult);
            }
            return $result;
        } else {
            return $this->searchAllServicesWithSupportedFields($fhirSearchParameters, $puuidBind);
        }
    }

    /**
     * Returns the Canonical URIs for the FHIR resource for each of the US Core Implementation Guide Profiles that the
     * resource implements.  Most resources have only one profile, but several like DiagnosticReport and Observation
     * has multiple profiles that must be conformed to.
     * @see https://www.hl7.org/fhir/us/core/CapabilityStatement-us-core-server.html for the list of profiles
     * @return string[]
     */
    public function getProfileURIs(): array
    {
        return $this->getProfileForVersions(self::USCGI_PROFILE_URI, $this->getSupportedVersions());
    }

    protected function getSupportedVersions(): array
    {
        return ['', '7.0.0', '8.0.0'];
    }
    /**
     * This method returns the FHIR search definition objects that are used to map FHIR search fields to OpenEMR fields.
     * Since the mapping can be one FHIR search object to many OpenEMR fields, we use the search definition objects.
     * Search fields can be combined as Composite fields and represent a host of search options.
     * @see https://www.hl7.org/fhir/search.html to see the types of search operations, and search types that are available
     * for use.
     * @return array
     */
    protected function loadSearchParameters(): array
    {
        return  [
            '_id' => new FhirSearchParameterDefinition(
                '_id',
                SearchFieldType::TOKEN,
                [new ServiceField('questionnaire_response_uuid', ServiceField::TYPE_UUID)]
            )
            ,'questionnaire' => new FhirSearchParameterDefinition(
                'questionnaire',
                SearchFieldType::REFERENCE,
                [new ServiceField('questionnaire_uuid', ServiceField::TYPE_UUID)]
            )
            ,'patient' => $this->getPatientContextSearchField()
            ,'authored' => new FhirSearchParameterDefinition(
                'authored',
                SearchFieldType::DATETIME,
                [new ServiceField('create_time', ServiceField::TYPE_STRING)]
            )
        ];
    }
}
